using System;

namespace _8052Sim
{
	/// <summary>
	/// Static object.  Cpu8052 is the object which represents the current state of the CPU.
	/// </summary>
	public class Cpu8052
	{
		/// <summary>
		/// Place the CPU in the freshly reset state.
		/// </summary>
		public static void HardReset()
		{
			MainForm.TerminalWindow.Clear();
			
			// randomize internal RAM
			for(int n=0;n<internalRam.Length;n++)
			{
				System.Random randy = new System.Random();
				randy.NextBytes(internalRam);
			}
			
			// Configure registers to reset values.
			Pc = 0;
			Dptr = 0;
			Acc = 0;
			B = 0;
			Sp = 0x07;
			Psw = 0;
			SetP(0,0xff);
			SetP(1,0xff);
			SetP(2,0xff);
			SetP(3,0xff);
			Ip = (byte)(Ip & 0xe0);
			Ie = (byte)(Ie & 0x60);
			Tcon = 0;
			Tmod = 0;
			Th0 = 0;
			Th1 = 0;
			Tl0 = 0;
			Tl1 = 0;
			Scon = 0;
			Pcon = (byte)(Pcon & 0x80);
			// Sbuf not initialized, intentionally.

			currentlyServicingAnInterrupt = false;
			wasInt0 = false;
			
			// start off by fetching an instruction to execute!			
			FetchNextInstruction();
		}

		
		/// <summary>
		/// Used for detecting when INT0 becomes active.
		/// </summary>
		private static bool wasInt0;
		

		/// <summary>
		/// Fetch a new instruction from the current PC.
		/// </summary>
		private static void FetchNextInstruction()
		{
			currentInstruction = InstructionInterpreter.FigureInstruction(
				Flash.GetByte((ushort)(Pc+0), Flash.A16),
				Flash.GetByte((ushort)(Pc+1), Flash.A16),
				Flash.GetByte((ushort)(Pc+2), Flash.A16)
			);
			nCrystalCyclesRemainingBeforeInstructionExecuted = currentInstruction.ExecutionTimeInCycles * 12;
			System.Diagnostics.Debug.Assert(nCrystalCyclesRemainingBeforeInstructionExecuted > 0);
		}
		

		/// <summary>
		/// Number of xtal cycles to wait before current instruction completes execution, 
		/// and a new instruction can be fetched.
		/// </summary>
		private static int nCrystalCyclesRemainingBeforeInstructionExecuted;


		/// <summary>
		/// Accumulator Register
		/// </summary>
		public static byte Acc
		{
			get	{	return sfrRegisters[0xE0];	}
			set	{	sfrRegisters[0xE0] = value;	}
		}
		
		
		/// <summary>
		/// B Register
		/// </summary>
		public static byte B
		{
			get	{	return sfrRegisters[0xF0];	}
			set	{	sfrRegisters[0xF0] = value;	}
		}
		
		
		/// <summary>
		/// Program Status Word Register
		/// </summary>
		public static byte Psw
		{
			get	{	return sfrRegisters[0xD0];	}
			set	{	sfrRegisters[0xD0] = value;	}
		}
		
		
		/// <summary>
		/// Program Status Word Register, Carry
		/// </summary>
		public static bool Psw_C
		{
			get	{	return ((sfrRegisters[0xD0] & 0x80) != 0);		}
			set	{	sfrRegisters[0xD0] = (byte)((sfrRegisters[0xD0] & (~0x80)) | (value ? 0x80 : 0x00));	}
		}
		

		/// <summary>
		/// Program Status Word Register, Overflow
		/// </summary>
		public static bool Psw_Ov
		{
			get	{	return ((sfrRegisters[0xD0] & 0x04) != 0);			}
			set	{	sfrRegisters[0xD0] |= (byte)((sfrRegisters[0xD0] & (~0x04)) | (value ? 0x04 : 0x00));	}
		}
		
		
		/// <summary>
		/// Program Status Word Register, Aux-Carry
		/// </summary>
		public static bool Psw_Ac
		{
			get	{	return ((sfrRegisters[0xD0] & 0x40) != 0);			}
			set	{	sfrRegisters[0xD0] |= (byte)((sfrRegisters[0xD0] & (~0x40)) | (value ? 0x40 : 0x00));	}
		}
		
		
		/// <summary>
		/// Interrupt Priority Register
		/// </summary>
		public static byte Ip
		{
			get	{	return sfrRegisters[0xB8];	}
			set	{	sfrRegisters[0xB8] = value;	}
		}
		
		
		/// <summary>
		/// Interrupt Enable Register
		/// </summary>
		public static byte Ie
		{
			get	{	return sfrRegisters[0xA8];	}
			set	{	sfrRegisters[0xA8] = value;	}
		}
		
		
		/// <summary>
		/// Timer Mode Register
		/// </summary>
		public static byte Tmod
		{
			get	{	return sfrRegisters[0x89];	}
			set	{	sfrRegisters[0x89] = value;	}
		}
		
		
		/// <summary>
		/// Timer Control Register
		/// </summary>
		public static byte Tcon
		{
			get	{	return sfrRegisters[0x88];	}
			set	{	sfrRegisters[0x88] = value;	}
		}
		
		
		/// <summary>
		/// Timer 0 High byte
		/// </summary>
		public static byte Th0
		{
			get	{	return sfrRegisters[0x8C];	}
			set	{	sfrRegisters[0x8C] = value;	}
		}
		
		
		/// <summary>
		/// Timer 0 Low byte
		/// </summary>
		public static byte Tl0
		{
			get	{	return sfrRegisters[0x8A];	}
			set	{	sfrRegisters[0x8A] = value;	}
		}
		
		
		/// <summary>
		/// Timer 1 High byte
		/// </summary>
		public static byte Th1
		{
			get	{	return sfrRegisters[0x8D];	}
			set	{	sfrRegisters[0x8D] = value;	}
		}
		
		
		/// <summary>
		/// Timer 1 Low byte
		/// </summary>
		public static byte Tl1
		{
			get	{	return sfrRegisters[0x8B];	}
			set	{	sfrRegisters[0x8B] = value;	}
		}
		
		
		/// <summary>
		/// Serial Control register
		/// </summary>
		public static byte Scon
		{
			get	{	return sfrRegisters[0x98];	}
			set	{	sfrRegisters[0x98] = value;	}
		}
		
		
		/// <summary>
		/// Serial Data Buffer Register
		/// </summary>
		public static byte Sbuf
		{
			get	{	return GetSfrRegister(0x99);	}
			set	{	SetSfrRegister(0x99, value);	}
		}


		/// <summary>
		/// Power Control Register
		/// </summary>
		public static byte Pcon
		{
			get	{	return sfrRegisters[0x87];	}
			set	{	sfrRegisters[0x87] = value;	}
		}
		
		
		/// <summary>
		/// Stack Pointer Register
		/// </summary>
		public static byte Sp
		{
			get	{	return sfrRegisters[0x81];	}
			set	{	sfrRegisters[0x81] = value;	}
		}
		
		
		/// <summary>
		/// DPH Register
		/// </summary>
		public static byte Dph
		{
			get	{	return sfrRegisters[0x83];	}
			set	{	sfrRegisters[0x83] = value;	}
		}
		
		
		/// <summary>
		/// DPL Register
		/// </summary>
		public static byte Dpl
		{
			get	{	return sfrRegisters[0x82];	}
			set	{	sfrRegisters[0x82] = value;	}
		}
		
		
		/// <summary>
		/// DPTR Register
		/// </summary>
		public static ushort Dptr
		{
			get	{	return (ushort)(((ushort)Dph)*256+Dpl);	}
			set	
			{
				Dpl = (byte)(value & 255);	
				Dph = (byte)(value >> 8);
			}
		}
		
		
		/// <summary>
		/// Program Counter Register
		/// </summary>
		public static ushort Pc
		{
			get	{	return pc;	}
			set	{	pc = value;	}
		}
		
		
		/// <summary>
		/// Write to the R0..R7 Registers
		/// </summary>
		public static void SetR(int index, byte val)
		{
			System.Diagnostics.Debug.Assert( (index >= 0) && (index <= 7) );
			SetInternalRam( ((Psw>>3)&3)*8+index, val);
		}
		
		
		/// <summary>
		/// Read the R0..R7 Registers
		/// </summary>
		public static byte GetR(int index)
		{
			System.Diagnostics.Debug.Assert( (index >= 0) && (index <= 7) );
			return internalRam[((Psw>>3)&3)*8+index];
		}
		
		
		/// <summary>
		/// Write to the P0..P3 Registers
		/// </summary>
		public static void SetP(int index, byte val)
		{
			System.Diagnostics.Debug.Assert( (index >= 0) && (index <= 3) );
			SetSfrRegister(0x80 + index * 0x10, val);
		}
		
		
		/// <summary>
		/// Read the P0..P3 Registers
		/// </summary>
		public static byte GetP(int index)
		{
			System.Diagnostics.Debug.Assert( (index >= 0) && (index <= 7) );
			return sfrRegisters[0x80 + index * 0x10];
		}
		
		
		/// <summary>
		/// Read a byte from the Internal RAM
		/// </summary>
		public static byte GetInternalRam(int index)
		{
			System.Diagnostics.Debug.Assert( (index >= 0) && (index <= 255) );
			return internalRam[index];
		}


		/// <summary>
		/// Write a byte to the Internal RAM
		/// </summary>
		public static void SetInternalRam(int index, byte val)
		{
			System.Diagnostics.Debug.Assert( (index >= 0) && (index <= 255) );
			internalRam[index] = val;
		}


		/// <summary>
		/// Read a byte from the Internal RAM
		/// </summary>
		public static byte GetDirect(int index)
		{
			if(index <= 0x7f)
				return GetInternalRam(index);
			return GetSfrRegister(index);
		}


		/// <summary>
		/// Write a byte to the Internal RAM
		/// </summary>
		public static void SetDirect(int index, byte val)
		{
			if(index <= 0x7f)
				SetInternalRam(index, val);
			else
				SetSfrRegister(index, val);
		}


		/// <summary>
		/// Write a byte to the Internal RAM
		/// </summary>
		public static byte GetIndirect(int index)
		{
			return GetInternalRam(index);
		}


		/// <summary>
		/// Write a byte to the Internal RAM
		/// </summary>
		public static void SetIndirect(int index, byte val)
		{
			SetInternalRam(index, val);
		}


		/// <summary>
		/// Calculate the raw flash address of the instruction which will execute,
		/// assuming "step-over" was requested.
		/// </summary>
		public static int AddressOverInstruction
		{
			get
			{
				IInstruction i = InstructionInterpreter.FigureInstruction(
					Flash.GetByte((ushort)(Pc+0), Flash.A16),
					Flash.GetByte((ushort)(Pc+1), Flash.A16),
					Flash.GetByte((ushort)(Pc+2), Flash.A16)
				);
				return i.AddressOverInstruction(
					Pc,
					Flash.GetByte((ushort)(Pc+0), Flash.A16),
					Flash.GetByte((ushort)(Pc+1), Flash.A16),
					Flash.GetByte((ushort)(Pc+2), Flash.A16)
				);
			}
		}
		
		
		/// <summary>
		/// Calculate the raw flash address of the instruction which will execute,
		/// assuming "step-into" was requested.
		/// </summary>
		public static int AddressIntoInstruction
		{
			get
			{
				IInstruction i = InstructionInterpreter.FigureInstruction(
					Flash.GetByte((ushort)(Pc+0), Flash.A16),
					Flash.GetByte((ushort)(Pc+1), Flash.A16),
					Flash.GetByte((ushort)(Pc+2), Flash.A16)
				);
				return i.AddressIntoInstruction(
					Pc,
					Flash.GetByte((ushort)(Pc+0), Flash.A16),
					Flash.GetByte((ushort)(Pc+1), Flash.A16),
					Flash.GetByte((ushort)(Pc+2), Flash.A16)
					);
			}
		}


		/// <summary>
		/// The 8052 chip executes another cycle (12 crystal cycles).
		/// </summary>
		public static bool MayBreak
		{
			get	{	return nCrystalCyclesRemainingBeforeInstructionExecuted == 1;	}
		}
		

		/// <summary>
		/// Used internally to communicate between timer 1 and serial transmission.
		/// </summary>
		private static bool timer1Overflowed = false;
		
		/// <summary>
		/// The 8052 chip executes another cycle (12 crystal cycles).
		/// </summary>
		public static void ExecuteOneCrystalCycle()
		{
//			System.Diagnostics.Debug.Assert(nCrystalCyclesRemainingBeforeInstructionExecuted > 0, "BLAH"+nCrystalCyclesRemainingBeforeInstructionExecuted.ToString() );
//			System.Diagnostics.Debugger.Log(0, "Cpu8052", "x");
			timer1Overflowed = false;
			
			// execute a cycle in the timer/counter module
			ExecuteOneCrystalCycleForTimerCounter(true);
			ExecuteOneCrystalCycleForTimerCounter(false);
			
			// update the serial device.
			ExecuteOneCrystalCycleForSerialTransmition();
			
			// execute a cycle in the instruction pipeline
			nCrystalCyclesRemainingBeforeInstructionExecuted--;
			if(nCrystalCyclesRemainingBeforeInstructionExecuted == 0)
			{
				// ready for a new instruction!!!
				ExecuteInstruction();	// complete the execution of this instruction (the necessary number of cycles have elapsed)

				// check if /INT0 has occurred
				if((!wasInt0) && Xilinx.INT0)
				{
					Tcon |= 0x01;	// set IE0 on inactive-to-active edge.
				}
				wasInt0 = Xilinx.INT0;
				
				// if it's time for an interrupt, allow it to alter 
				// the normal flow of the code by involking the correct ISR!
				if( (Ie & 0x80) != 0 )					// Master Enable
					if(!currentlyServicingAnInterrupt)	// only if not already handling an interrupt... (use RETI to clear this)
						HandlePendingInterrupt();
				
				FetchNextInstruction();	// fetch the new instruction, and begin waiting for the correct # of cycles.
			}
		}
		

		/// <summary>
		/// Only called if the master interrupt enable is active (EA).
		/// This routine looks for an interrupt to vector to, *if* necessary.
		/// It will obey the priority system.
		/// </summary>
		private static void HandlePendingInterrupt()
		{
			// order of scanning interrupts...
			// IE0, TF0, IE1, TF1, RI&TI
			// First, scan interrupts with their priority bit set to HIGH, then scan interrupts who are set to LOW priority.
			// Stop as soon as an interrupt is found that needs servicing.
			
			for(int pri = 1; pri >= 0; pri--)
			{
				// first, check the interrupts at priority 1
				if( ((Ip & 0x01) == ((pri==1)?0x01:0)) && ((Ie & 0x01) != 0) && ((Tcon & 0x01) != 0) )	// IE0
				{
					Cpu8052.PushStack(Pc);
					Pc = 0x0003;
					Tcon &= 0xff - 0x01;	// clear the IE0 flag, since we just vectored to handle it!
					currentlyServicingAnInterrupt = true;
				}
				else if( ((Ip & 0x02) == ((pri==1)?0x02:0)) && ((Ie & 0x02) != 0) && ((Tcon & 0x20) != 0) )	// TF0
				{
					Cpu8052.PushStack(Pc);
					Pc = 0x000b;
					Tcon &= 0xff - 0x20;	// clear the TF0 flag, since we just vectored to handle it!
					currentlyServicingAnInterrupt = true;
				}
				else if( ((Ip & 0x04) == ((pri==1)?0x04:0)) && ((Ie & 0x04) != 0) && ((Tcon & 0x04) != 0) )	// IE1
				{
					Cpu8052.PushStack(Pc);
					Pc = 0x0013;
					Tcon &= 0xff - 0x04;	// clear the IE1 flag, since we just vectored to handle it!
					currentlyServicingAnInterrupt = true;
				}
				else if( ((Ip & 0x08) == ((pri==1)?0x08:0)) && ((Ie & 0x08) != 0) && ((Tcon & 0x80) != 0) )	// TF1
				{
					Cpu8052.PushStack(Pc);
					Pc = 0x001b;
					Tcon &= 0xff - 0x80;	// clear the TF0 flag, since we just vectored to handle it!
					currentlyServicingAnInterrupt = true;
				}
				else if( ((Ip & 0x10) == ((pri==1)?0x10:0)) && ((Ie & 0x10) != 0) && ((Scon & 0x03) != 0) )	// RI&TI
				{
					Cpu8052.PushStack(Pc);
					Pc = 0x0023;
					currentlyServicingAnInterrupt = true;
					// RI or TI must be cleared manually by the program before executing a RETI, 
					// or another interrupt occurs immediately.
				}

				// if we found an interrupt to service at this priority, stop looking for more!
				if(currentlyServicingAnInterrupt)
					break;
			}
		}

		/// <summary>
		/// Used by interrupt handling system to know whether it's 
		/// already busy with an interrupt handler.
		/// </summary>
		private static bool currentlyServicingAnInterrupt = false;

		/// <summary>
		/// Called by the RETI instruction, when it executes.
		/// This tells the CPU it's ok to service another interrupt if requested!
		/// </summary>
		public static void InterruptServiceComplete()
		{
			currentlyServicingAnInterrupt = false;
		}
		
		/// <summary>
		/// Used to implement the internal divide-by-12 stage of the timer/counter circuit.
		/// </summary>
		private static int timer0OscilatorDivider = 0;


		/// <summary>
		/// Used to implement the internal divide-by-12 stage of the timer/counter circuit.
		/// </summary>
		private static int timer1OscilatorDivider = 0;


		/// <summary>
		/// Used to detect high-to-low transitions of this pin.
		/// </summary>
		private static bool wasT0State = true;


		/// <summary>
		/// Used to detect high-to-low transitions of this pin.
		/// </summary>
		private static bool wasT1State = true;


		/// <summary>
		/// This logic executes the logic which is rendered in 
		/// figure 2.11, page 30 of "The 8051 Microcontroller" book.
		/// </summary>
		private static void ExecuteOneCrystalCycleForTimerCounter(bool isTimer0)
		{
			if((Cpu8052.Tcon & (isTimer0 ? 0x10 : 0x40)) != 0)	// run?
			{	
				if( 
					(!((Cpu8052.Tmod & (isTimer0 ? 0x08 : 0x80)) != 0))	||	// Gate
					((Cpu8052.GetP(3) & (isTimer0 ? 0x04 : 0x08)) != 0)		// /INT
				)
				{
					if((Cpu8052.Tmod & (isTimer0 ? 0x04 : 0x40)) != 0)	// timer or counter?
					{
						// C/T = 1 -- Feed timer using input pin.
						// NOTE:  Read Page 32 for details about 8051 exact sampling methods.
						// The following implementation is simplified, but should not 
						// cause a problem.
						// 8051's cound high-to-low transitions.
						if(	
							(isTimer0 ? wasT0State : wasT1State) && 
							((Cpu8052.GetP(3) & (isTimer0 ? 16 : 32) ) == 0) 
						)
						{
							TimerTick(isTimer0);
						}
					}
					else
					{
						// C/T = 0 -- feed timer using xtal.
						if(isTimer0)
							timer0OscilatorDivider++;
						else
							timer1OscilatorDivider++;
							
						if( (isTimer0 ? timer0OscilatorDivider : timer1OscilatorDivider) == 24)
						{	
							if(isTimer0)
								timer0OscilatorDivider = 0;
							else
								timer1OscilatorDivider = 0;
							TimerTick(isTimer0);
						}
					}
					if(isTimer0)
						wasT0State = (Cpu8052.GetP(3) & 16) != 0;
					else
						wasT1State = (Cpu8052.GetP(3) & 32) != 0;
				}
			}
		}
		
		
		/// <summary>
		/// A timer tick needs to be counted.
		/// </summary>
		private static void TimerTick(bool isTimer0)
		{
			switch(isTimer0 ? (Cpu8052.Tmod & 0x03) : ((Cpu8052.Tmod>>4) & 0x03))
			{
				case 0:	TimerTickMode0(isTimer0);	return;
				case 1:	TimerTickMode1(isTimer0);	return;
				case 2:	TimerTickMode2(isTimer0);	return;
				case 3:	TimerTickMode3(isTimer0);	return;
				default:
					System.Diagnostics.Debug.Assert(false);
					break;
			}
		}
		
		
		/// <summary>
		/// A timer tick needs to be counted, using Mode0.
		/// See Pg. 31
		/// </summary>
		private static void TimerTickMode0(bool isTimer0)
		{
			if(isTimer0)
				Cpu8052.Th0 = (byte)((Cpu8052.Th0 + 1) & 0x1f);
			else
				Cpu8052.Th1 = (byte)((Cpu8052.Th1 + 1) & 0x1f);
			
			if( (isTimer0 ? Cpu8052.Th0 : Cpu8052.Th1) == 0x00 )
			{
				if(isTimer0)
					Cpu8052.Tl0++;
				else
					Cpu8052.Tl1++;
				
				if( (isTimer0 ? Cpu8052.Tl0 : Cpu8052.Tl1) == 0x00 )
					TimerOverflow(isTimer0);
			}
		}
		
		
		/// <summary>
		/// A timer tick needs to be counted, using Mode1.
		/// See Pg. 31
		/// </summary>
		private static void TimerTickMode1(bool isTimer0)
		{
			if(isTimer0)
				Cpu8052.Th0++;
			else
				Cpu8052.Th1++;
			
			if( (isTimer0 ? Cpu8052.Th0 : Cpu8052.Th1) == 0x00 )
			{
				if(isTimer0)
					Cpu8052.Tl0++;
				else
					Cpu8052.Tl1++;
				
				if( (isTimer0 ? Cpu8052.Tl0 : Cpu8052.Tl1) == 0x00 )
					TimerOverflow(isTimer0);
			}
		}
		
		
		/// <summary>
		/// A timer tick needs to be counted, using Mode2.
		/// See Pg. 31
		/// </summary>
		private static void TimerTickMode2(bool isTimer0)
		{
			if(isTimer0)
				Cpu8052.Tl0++;
			else
				Cpu8052.Tl1++;
			
			if( (isTimer0 ? Cpu8052.Tl0 : Cpu8052.Tl1) == 0x00 )
			{
				if(isTimer0)
					Cpu8052.Tl0 = Cpu8052.Th0;
				else
					Cpu8052.Tl1 = Cpu8052.Th1;
				
				TimerOverflow(isTimer0);
			}
		}

		/// <summary>
		/// A timer tick needs to be counted, using Mode3.
		/// See Pg. 31
		/// </summary>
		private static void TimerTickMode3(bool isTimer0)
		{
			if(isTimer0)
			{
				Cpu8052.Tl0++;
				if(Cpu8052.Tl0 == 0x00)
					TimerOverflow(true);
				Cpu8052.Th0++;
				if(Cpu8052.Th0 == 0x00)
					TimerOverflow(false);
			}
			else
			{
				Cpu8052.Tl1++;
				if(Cpu8052.Tl1 == 0x00)
					TimerOverflow(true);
				Cpu8052.Th1++;
				if(Cpu8052.Th1 == 0x00)
					TimerOverflow(false);
			}
		}
		

		/// <summary>
		/// Allows for a common way to signal the interrupt, and to signal the serial 
		/// code that timer1 has overflowed.
		/// </summary>
		private static void TimerOverflow(bool isTimer0)
		{
			Cpu8052.Tcon |= (byte)(isTimer0 ? 0x20 : 0x80);	// trigger interrupt!

			// if timer1 overflowed, make sure the serial code knows!
			if(isTimer0 == false)
				timer1Overflowed = true;
		}	
		
		
		/// <summary>
		/// Cause the external state of the 8052 to represent the completed instruction.
		/// </summary>
		private static void ExecuteInstruction()
		{
//			System.Diagnostics.Debugger.Log(0, "Cpu8052", "I\n");
			currentInstruction.Execute(
				Pc,
				Flash.GetByte((ushort)(Pc+0), Flash.A16),
				Flash.GetByte((ushort)(Pc+1), Flash.A16),
				Flash.GetByte((ushort)(Pc+2), Flash.A16)
			);
		}


		/// <summary>
		/// Get a value from inside the stack, without affecting the stack.
		/// eg.  getting [-1]*256+[0] == a return-address
		/// SP always points at the last item pushed on the stack.
		/// </summary>
		public static byte Stack(int offset)
		{
			return internalRam[Sp+offset];
		}

		
		/// <summary>
		/// Remove and return a byte from the stack.
		/// </summary>
		public static byte PopStack()
		{
			return internalRam[Sp--];
		}
		
		
		/// <summary>
		/// Add a byte to the stack.
		/// </summary>
		public static void PushStack(byte val)
		{
			internalRam[++Sp] = val;
		}
		
		
		/// <summary>
		/// Add a return-address to the stack. (Push LOWBYTE, then Push HIBYTE), like ACALL
		/// </summary>
		public static void PushStack(ushort val)
		{
			PushStack((byte)(val & 0xff));
			PushStack((byte)(val >> 8));
		}
		
		
		/// <summary>
		/// Set the value of a bit in internal ram or SFR Register
		/// </summary>
		public static void SetBit(byte bitNumber, bool state)
		{
			if(bitNumber<=0x7f)
			{
				// bit-address of bit memory.
				if(state == true)
				{
					SetInternalRam(
						0x20 + bitNumber / 8,  
						(byte)(GetInternalRam(0x20 + bitNumber / 8) | (byte)(1 << (bitNumber & 7)))
					);
				}
				else
				{
					SetInternalRam(
						0x20 + bitNumber / 8,
						(byte)(GetInternalRam(0x20 + bitNumber / 8) & (byte)(~(1 << (bitNumber & 7))))
					);
				}
			}
			else
			{
				// bit-address of SFR!
				if(state == true)
				{
					SetSfrRegister(
						bitNumber & 0xf8,
						(byte)(GetSfrRegister(bitNumber & 0xf8) | (byte)(1 << (bitNumber & 7)))
					);
				}
				else
				{
					SetSfrRegister(
						bitNumber & 0xf8,
						(byte)(GetSfrRegister(bitNumber & 0xf8) & (byte)(~(1 << (bitNumber & 7))))
					);
				}
			}
		}
		
/*
		/// <summary>
		/// Get the value of an SFR Register Bit
		/// </summary>
		public static void GetSfrBit(int index)
		{
			System.Diagnostics.Debug.Assert( (index >= 128) && (index <= 255) );
			switch(index)
			{
				case 0x90:
					// the I2C pins may be pulled low by an external device...
					// emulate this by masking-out bits which are pulled low.
					byte val = sfrRegisters[index];
					if(!Sta013.SdaState())	val &= (byte)(0xfe);
					if(!Sta013.SclState())	val &= (byte)(0xfd);
					return val;
			}		
		}
		
		
		/// <summary>
		/// Set the value of an SFR Register Bit
		/// </summary>
		public static void SetSfrBit(int index)
		{
		
		}
*/
		
		/// <summary>
		/// Get the value of an SFR Register
		/// </summary>
		public static byte GetSfrRegister(int index)
		{
			System.Diagnostics.Debug.Assert( (index >= 128) && (index <= 255) );
			switch(index)
			{
				// If reading SBUF, the last recieved character is returned.
				// in my sim, it always returns the next character which is queue'd.
				case 0x99:
					if(MainForm.TerminalWindow.HasCharacter())
						return (byte)MainForm.TerminalWindow.GetCharacter();
					break;
				
				// update the RI flag if reading SCON
				case 0x98:
					sfrRegisters[index] = (byte)( (sfrRegisters[index] & ~0x01) | (MainForm.TerminalWindow.HasCharacter() ? 0x01 : 0x00) );	// set RI flag!
					break;
			}
			return sfrRegisters[index];
		}
		
		
		/// <summary>
		/// Set the value of an SFR Register
		/// </summary>
		public static void SetSfrRegister(int index, byte val)
		{
			System.Diagnostics.Debug.Assert( (index >= 128) && (index <= 255) );
			sfrRegisters[index] = val;
			
			switch(index)
			{
				case 0x90:	// P1
					// handle the STA013 I2C Interface
					Sta013.I2CState(
						(sfrRegisters[index] & 0x01) != 0,	// I2C CLK
						(sfrRegisters[index] & 0x02) != 0	// I2C DATA
					);

					// handle the External Hardware Reset line.
					if( (sfrRegisters[index] & 0x04) == 0)
					{
						Xilinx.HardReset();
						Sta013.HardReset();
					}
					
					// handle the Xilinx communication interface.
					Xilinx.Data((sfrRegisters[index] & 0x80) != 0);
					Xilinx.Clock((sfrRegisters[index] & 0x20) != 0);
					break;
				
				case 0x99:	// SBUF
					BeginSerialTransmition();
					break;

				case 0xB0:	// P3
					Flash.A16 = ((val & 16) != 0) ? true : false;
//					MainForm.LockSourceView();
					MainForm.CurrentSegment.BackColor = 
						Flash.A16 ? System.Drawing.Color.LightGreen : System.Drawing.Color.LightSalmon;
//					MainForm.UnlockSourceView();
					break;
			}
		}
		
		
		/// <summary>
		/// Get the value of a bit from internal ram or SFR Register.
		/// NOTE:  The result of this call may reflect the state of an
		/// external pin being pulled low, as opposed to the internal 
		/// latch state.  Therefor it is not possible to use this routine
		/// to perform a read-modify-write.
		/// </summary>
		public static bool GetBit(byte bitNumber)
		{
			if(bitNumber<=0x7f)
			{
				// bit-address of bit memory.
				return ((internalRam[0x20 + bitNumber / 8] & (byte)(1 << (bitNumber & 7))) != 0) ? true : false;
			}
			else
			{
				switch(bitNumber)
				{
					case 0x91:
						bool val = ((GetSfrRegister(bitNumber & 0xf8) & (byte)(1 << (bitNumber & 7))) != 0) ? true : false;
						return Sta013.I2CAck(val);	// allow STA013 to pull the bit low.
						
					default:
						// bit-address of SFR!
						return ((GetSfrRegister(bitNumber & 0xf8) & (byte)(1 << (bitNumber & 7))) != 0) ? true : false;
				}
			}
		}


		/// <summary>
		/// As a byte of data is being transmitted, the serial hardware passes through these states.
		/// </summary>
		private enum SerialTransmitStateMode1
		{
			SendingStartBit,	// Transmitter spends one baud clock here.
			SendingDataBit0,	// Transmitter spends one baud clock here.
			SendingDataBit1,	// Transmitter spends one baud clock here.
			SendingDataBit2,	// Transmitter spends one baud clock here.
			SendingDataBit3,	// Transmitter spends one baud clock here.
			SendingDataBit4,	// Transmitter spends one baud clock here.
			SendingDataBit5,	// Transmitter spends one baud clock here.
			SendingDataBit6,	// Transmitter spends one baud clock here.
			SendingDataBit7,	// Transmitter spends one baud clock here.
			SendingStopBit,		// Transmitter spends one baud clock here.
			Done,				// Set TI
		};
		
		
		/// <summary>
		/// Remember the current state of the serial transmition hardware.
		/// </summary>
		private static SerialTransmitStateMode1 serialTransmitState = SerialTransmitStateMode1.Done;
		
		
		/// <summary>
		/// Begin a new transmission.  This is called whenever someone writes to SBUF.
		/// </summary>
		private static void BeginSerialTransmition()
		{
			switch((Cpu8052.Scon>>6)&0x03)	// serial port mode
			{
				case 0:	break;	// serial xmit not emulated.
					
				case 1:			// pg. 36
					serialTransmitState = SerialTransmitStateMode1.SendingStartBit;
					serialXmitClockDivider = 0;
//					System.Diagnostics.Debugger.Log(0, "XMIT", "Start: "+Simulation.CycleNumber.ToString()+"\n");
					break;
					
				case 2:	break;	// serial xmit not emulated.
				case 3:	break;	// serial xmit not emulated.
			}
		}
		

		/// <summary>
		/// divider, used by serial transmit code.
		/// </summary>
		private static int serialXmitClockDivider = 0;

		
		/// <summary>
		/// Serial transmition is implemented by this code.
		/// </summary>
		private static void ExecuteOneCrystalCycleForSerialTransmition()
		{
			switch((Cpu8052.Scon>>6)&0x03)	// serial port mode
			{
				case 0:	break;	// serial xmit not emulated.
					
				case 1:			// pg. 36
					if(serialTransmitState != SerialTransmitStateMode1.Done)
					{
						if(timer1Overflowed)
						{
							serialXmitClockDivider++;
							if(serialXmitClockDivider >= (((Pcon & 0x80) != 0) ? 16 : 32))
							{
//								System.Diagnostics.Debugger.Log(0, "XMIT", "BIT: "+Simulation.CycleNumber.ToString()+"\n");
								serialXmitClockDivider = 0;
								serialTransmitState++;
								if(serialTransmitState == SerialTransmitStateMode1.Done)
								{
//									System.Diagnostics.Debugger.Log(0, "XMIT", "DONE: "+Simulation.CycleNumber.ToString()+"\n");
//									System.Diagnostics.Debugger.Log(0, "XMIT", "BYTE: "+sfrRegisters[0x99]+"\n");
									DisplayCharacter((char)sfrRegisters[0x99]);	// MUST NOT TRIGGER SENDING/RECIEVING THIS BYTE AGAIN!  use direct-access to value!
									Scon |= 0x02;	// set TI!
								}
							}
						}
					}
					break;
				
				case 2:	break;	// serial xmit not emulated.
				case 3:	break;	// serial xmit not emulated.
			}
		}
		
		
		/// <summary>
		/// This is called by the serial port output code in Cpu8052, when characters are
		/// transmitted bye the CPU.
		/// </summary>
		public static void DisplayCharacter(char c)
		{
			MainForm.TerminalWindow.DisplayCharacter(c);
		}
		

		/// <summary>
		/// The instruction which the cpu is currently working on.
		/// </summary>
		private static IInstruction currentInstruction;
		
		
		/// <summary>
		/// The actual storage location for Internal RAM
		/// </summary>
		private static byte[] internalRam = new byte[256];


		/// <summary>
		/// Storage space for SFR Registers...  This memory-block is sparsely populated.
		/// </summary>
		private static byte[] sfrRegisters = new byte[256];


		/// <summary>
		/// The current Program Counter value, since it is not represented within the Internal RAM.
		/// </summary>
		private static ushort pc;	// program counter has no internal address.
	}
}
