using System;

namespace _8052Sim
{
	/// <summary>
	/// Base Class for all Instruction Objects.  
	/// InstructionBase provides common implementations for tasks which 
	/// may be performed by many instructions.
	/// </summary>
	public abstract class InstructionBase : IInstruction
	{
		/// <summary>
		/// The general classification an instruction falls into.
		/// </summary>
		public enum InstructionType
		{
			IT_Arithmetic,
			IT_Logic,
			IT_DataMove,
			IT_Call,
			IT_Jump,
			IT_Boolean
		}
		
		/// <summary>
		/// Constructor.
		/// </summary>
		protected InstructionBase(
			InstructionBase.InstructionType _type,
			String _name,
			int _nBytes,
			uint val,
			uint mask,
			byte _executionTimeInCycles)
		{
			type = _type;
			name = _name;
			nBytes = (ushort)_nBytes;
			instructionWordValue = val;
			instructionWordMask = mask;
			executionTimeInCycles = _executionTimeInCycles;
		}

		// ----------------------------------------------------------------
		// for IInstruction
		// ----------------------------------------------------------------
		/// <summary>
		/// Override as necessary!
		/// By default, this routine will calculate the step-over as 
		/// the next instruction after this one.
		/// </summary>
		public virtual ushort AddressOverInstruction(ushort address, byte b1, byte b2, byte b3)
		{
			switch(type)
			{
				case InstructionType.IT_Arithmetic:
				case InstructionType.IT_Logic:
				case InstructionType.IT_DataMove:
				case InstructionType.IT_Boolean:
				case InstructionType.IT_Call:
				case InstructionType.IT_Jump:
					return (ushort)(address+nBytes);
					
				default:
					System.Diagnostics.Debug.Assert(false);
					break;
			}
			return 0;
		}

		/// <summary>
		/// Override as necessary!
		/// By default, this routine will act the same as "step-over".
		/// </summary>
		public virtual ushort AddressIntoInstruction(ushort address, byte b1, byte b2, byte b3)
		{
			return AddressOverInstruction(address, b1, b2, b3);
		}

		/// <summary>
		/// Returns TRUE if this instruction object is able to interpret the provided bytes.
		/// </summary>
		public bool Detect(byte b1, byte b2, byte b3)
		{
			uint val;
			switch(nBytes)
			{
				case 1:
					val = (uint)b1;
					break;
					
				case 2:
					val = ((uint)b1<<8) + (uint)b2;
					break;

				case 3:
					val = ((uint)b1<<16) + ((uint)b2 << 8) + (uint)b3;
					break;

				default:
					System.Diagnostics.Debug.Assert(false);
					return false;
			}
			return (val & instructionWordMask) == instructionWordValue;
		}


		/// <summary>
		/// Performs the actions on the CPU which implement this instruction.
		/// </summary>
		public abstract void Execute(ushort address, byte b1, byte b2, byte b3);
		

		/// <summary>
		/// The number of CPU cycles this instruction requires.
		/// </summary>
		public int ExecutionTimeInCycles
		{
			get
			{
				return executionTimeInCycles;
			}
		}

		/// <summary>
		/// Returns the number of bytes this instruction occupies (1..3)
		/// </summary>
		public ushort SizeInBytes
		{
			get
			{
				return nBytes;
			}
		}


		/// <summary>
		/// Sets the Carry, Aux-Carry, and Overflow flags for you.  
		/// Returns the result of the addition.
		/// </summary>
		protected byte AddAndSetFlags(byte v1, byte v2, bool cFlag)
		{
			// decide on a numeric value for the carry flag...
			int c = cFlag ? 1 : 0;
			
			// Set Carry
			Cpu8052.Psw_C = ( ((int)v1 + (int)v2 + c ) > 0xff );
			
			// set Aux-Carry
			Cpu8052.Psw_Ac = ( ((v1 & 0x0f) + (v2 & 0x0f) + c) > 0x0f );
			
			// set Overflow
			if( ((v1 & 0x7f) + (v2 & 0x7f) + c) > 0x7f )
				Cpu8052.Psw_Ov = (!Cpu8052.Psw_C);
			else
				Cpu8052.Psw_Ov = Cpu8052.Psw_C;
				
			return (byte)((int)v1 + (int)v2 + c);
		}


		/// <summary>
		/// Sets the Carry, Aux-Carry, and Overflow flags for you.  
		/// Returns the result of the subtraction.
		/// </summary>
		protected byte SubtractAndSetFlags(byte v1, byte v2, bool cFlag)
		{
			// decide on a numeric value for the carry flag...
			int c = cFlag ? 1 : 0;
			
			// Set Carry
			Cpu8052.Psw_C = ( ((int)v1 - (int)v2 - c ) < 0);
			
			// set Aux-Carry
			Cpu8052.Psw_Ac = ( ((v1 & 0x0f) - (v2 & 0x0f) - c) < 0);
			
			// set Overflow
			if( ((v1 & 0x7f) - (v2 & 0x7f) - c) < 0)
				Cpu8052.Psw_Ov = (!Cpu8052.Psw_C);
			else
				Cpu8052.Psw_Ov = Cpu8052.Psw_C;
				
			return (byte)((int)v1 - (int)v2 - c);
		}


		/// <summary>
		/// Default operation is to return the name of the instruction.
		/// Override as needed, if the instruction needs disassembled 
		/// in a more complicated way.
		/// </summary>
		public String ToString(ushort address, byte b1, byte b2, byte b3)
		{
			return name;
		}


		/// <summary>
		/// The name of this instruction
		/// </summary>
		private String name;	// the name of the instruction		

		/// <summary>
		/// The number of bytes this instruction is
		/// </summary>
		private ushort nBytes;	// number of bytes making up this instruction.

		/// <summary>
		/// The general type this instruction is
		/// </summary>
		InstructionBase.InstructionType type;
		
		/// <summary>
		/// The 3 bytes which make up the instruction are first masked, then compared 
		/// against this value to detect if the instruction is recognized by this object.
		/// 0x00aabbcc
		/// 0x0000aabb
		/// 0x000000aa
		/// </summary>
		private uint instructionWordValue;	// the value portion of the instruction.

		/// <summary>
		/// The 3 bytes which make up the instruction are first masked, then compared 
		/// against a value to detect if the instruction is recognized by this object.
		/// </summary>
		private uint instructionWordMask;	// the mask which is used before comparing with the value.
		
		/// <summary>
		/// The number of CPU cycles that are required before this instruction can complete.
		/// </summary>
		private byte executionTimeInCycles;
	}
}
