using System;

namespace _8052Sim
{
	/// <summary>
	/// A CachedFile is a file which is stored in FileCache.
	/// It has been read from disk and is now stored as an array of strings, one
	/// string for each line of the file.
	/// </summary>
	public class CachedFile
	{
		/// <summary>
		/// Cache a new file into an array of Strings, one for each line.
		/// </summary>
		public CachedFile(String fn)
		{
			filename = fn;

			System.IO.StreamReader reader = System.IO.File.OpenText("C:\\MP3PlayerHardware\\Source\\"+filename);
			while (reader.Peek() > -1) 
				lines.Add(reader.ReadLine());
			reader.Close();

			addressOfLine = new System.Collections.ArrayList();
			for(int n=0;n<lines.Count;n++)
				addressOfLine.Add(null);

			segmentOfLine = new System.Collections.ArrayList();
			for(int n=0;n<lines.Count;n++)
				segmentOfLine.Add(null);
		}
		
		/// <summary>
		/// Remember the address of a line of the file.
		/// Lines are numbered 1 through n.
		/// </summary>
		public void AssignAddressOfLine(int line, ushort addr, bool a16)
		{
			System.Diagnostics.Debug.Assert( 
				(addressOfLine[line-1] == null) || 
				((ushort)addressOfLine[line-1] == addr) 
			);
			addressOfLine[line-1] = addr;
			if(segmentOfLine[line-1] == null)
				segmentOfLine[line-1] = (byte)0;
			segmentOfLine[line-1] = (byte)((byte)(segmentOfLine[line-1]) | (1<<(a16?1:0)));
		}
		
		/// <summary>
		/// Remember the address of a line of the file.
		/// Lines are numbered 1 through n.
		/// </summary>
		public ushort AddressOfLine(int line)
		{
			if(addressOfLine[line-1] == null)
			{
				if(line < addressOfLine.Count)
					return AddressOfLine(line+1);
				return 0x0000;
			}
			return (ushort)addressOfLine[line-1];
		}
		
		public bool SegmentOfLine(int line)
		{
			if(segmentOfLine[line-1] == null)
			{
				if(line < segmentOfLine.Count)
					return SegmentOfLine(line+1);
				return false;
			}
			return (((byte)segmentOfLine[line-1]) & (byte)0x02) != 0;
		}
		
		/// <summary>
		/// Discover the filename of this cached file.
		/// </summary>
		public String Filename
		{
			get
			{
				return filename;
			}
		}
		
		/// <summary>
		/// The lines of text which were cached.
		/// </summary>
		public String[] Lines
		{
			get
			{
				String[] theLines = new String[lines.Count];
				for(int n=0;n<lines.Count;n++)
					theLines[n] = (String)lines[n];
				return theLines;
			}
		}
		
		/// <summary>
		/// Returns TRUE if the filename & segment match
		/// </summary>
		public bool Compare(String fn)
		{
			return (String.Compare(fn, filename) == 0);
		}
		
		
		/// <summary>
		/// the filename of this cached file.
		/// </summary>
		private String filename;

		/// <summary>
		/// the lines of text which were cached.
		/// </summary>
		private System.Collections.ArrayList lines = new System.Collections.ArrayList();	// array of String's
		System.Collections.ArrayList addressOfLine = null;	// ushort's
		System.Collections.ArrayList segmentOfLine = null;	// bit0 = segment0, bit1 = segment1
	}
}
