using System;

namespace _8052Sim
{
	/// <summary>
	/// Static object.  This is the CODE ROM memory space, accessed via MOVC instructions.
	/// </summary>
	public class Flash
	{
		/// <summary>
		/// Flash object Constructor, call when you wish for the Flash to be burned.
		/// </summary>
		public static void Startup()
		{
			// ensure the entire Flash chip starts out fully erased.
			for(int n=0;n<65536*2;n++)
				code[n] = new FlashByte();

			// build the collection of array-lists which hold the breakpoints.				
			for(int n=0;n<65536*2;n++)
				breakpoints[n] = new System.Collections.ArrayList();

			// Scan the Map files of the two parts that make up the rom
			A16 = false;
			ScanMapFile(MainForm.SourceFileMenu, "Code\\Main\\Segment0\\Main0.map");
			currentFilename = SlashToBackslash("Code/PaulMon2/pm2_mp3.rst");
			ScanRstFile(
				FileCache.TheCachedFile(currentFilename)
			);
			
			A16 = true;
			ScanMapFile(MainForm.SourceFileMenu, "Code\\Main\\Segment1\\Main1.map");
			currentFilename = SlashToBackslash("Code/PaulMon2/pm2_mp3.rst");
			ScanRstFile(
				FileCache.TheCachedFile(currentFilename)
			);
			
			AddFileToSourceMenu(SlashToBackslash(currentFilename));
			
			// finally, scan the hex file itself, and verify the stored data 
			// is correct.  Also, pick up any additional bytes which are present, 
			// but do not have source files (library routines).
			ScanHexFile("complete.hex");
		}

		private const String dec =				"[0-9]+";							// line number contained in rst file
		private const String module =			"[a-zA-Z0-9]+";						// name of a module
		private const String symbol =			"[a-zA-Z0-9_]+";					// name of a symbol
		private const String segmentName =		"[a-zA-Z0-9_]+";					// name of a segment
		private const String segmentModes =		"[a-zA-Z0-9_,]+";					// modes of a segment, like ABS, REL, CON, etc.
		private const String hex =				"[0-9a-fA-F]";						// f
		private const String hex8 =				hex+"{2}";							// f0
		private const String hex16 =			hex+"{4}";							// f043
		private const String formattedHex16 =	"(0x|\\$)"+hex16;					// 0xf043
		private const String formattedHex =		"(0x|\\$)"+hex+"+";					// 0xf5
		private const String textLabel =		"[a-zA-Z0-9_]+";					// a label name
		private const String localLabel =		"[0-9]+[\\$]";						// a local label 0004$
		private const String label =			"("+localLabel+"|"+textLabel+")";	// any label
		private const String scope =			"[:][:]?";							// scope of a label (: or ::)
		private const String hexOrDec =			"("+formattedHex+"|"+dec+")";		// "[0-9a-fA-Fx]+";			// any integral count
		private const String comment =			";.*";								// how a comment looks...
								
		// identify a line of data in a .HEX file.
		private static System.Text.RegularExpressions.Regex regexScanMapFileModule = 
			new System.Text.RegularExpressions.Regex( 
				"^(?<file>[\\w\\./\\\\]+).rel[\t]\\[\\s+\\w*\\s*\\]$", 
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
		
		private static System.Text.RegularExpressions.Regex regexScanMapFileXSEG = 
			new System.Text.RegularExpressions.Regex( 
				"^(?<segment>.SEG)\\s+(?<startAddress>\\w\\w\\w\\w)\\s+(?<size>\\w\\w\\w\\w)\\s+=", 
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);

		private static System.Text.RegularExpressions.Regex[] regexScanMapFileSymbol = new System.Text.RegularExpressions.Regex[4]
		{
			// BSEG
			new System.Text.RegularExpressions.Regex( 
				"^\\s+" + "0B:" + "(?<addr>\\w\\w\\w\\w)\\s+(?<symbol>\\w+)",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			),
			// CSEG
			new System.Text.RegularExpressions.Regex( 
				"^\\s+" + "0C:" + "(?<addr>\\w\\w\\w\\w)\\s+(?<symbol>\\w+)",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			),
			// DSEG
			new System.Text.RegularExpressions.Regex( 
				"^\\s+" + "" + "(?<addr>\\w\\w\\w\\w)\\s+(?<symbol>\\w+)",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			),
			// XSEG
			new System.Text.RegularExpressions.Regex( 
				"^\\s+" + "0D:" + "(?<addr>\\w\\w\\w\\w)\\s+(?<symbol>\\w+)",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			),
		};
		
		private static System.Text.RegularExpressions.Regex regexScanMapFileHexadecimal = 
			new System.Text.RegularExpressions.Regex( 
				"^Hexadecimal",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);

		private static System.Text.RegularExpressions.Regex regexParseModule = 
			new System.Text.RegularExpressions.Regex( 
				"^[ ]{25}[ ]{0,5}(?<lineNumber>"+dec+")[ \t]+.module[ \t]+(?<module>"+module+")$",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);

		private static System.Text.RegularExpressions.Regex regexParseGlobl = 
			new System.Text.RegularExpressions.Regex( 
				"^[ ]{25}[ ]{0,5}(?<lineNumber>"+dec+")[ \t]+.globl[ \t]+(?<symbol>"+symbol+")$",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);

		private static System.Text.RegularExpressions.Regex regexParseArea = 
			new System.Text.RegularExpressions.Regex( 
				"^[ ]{25}[ ]{0,5}(?<lineNumber>"+dec+")[ \t]+.area[ \t]+(?<segmentName>"+segmentName+")[ \t]+[(](?<segmentModes>"+segmentModes+")[)]$",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
		
		private static System.Text.RegularExpressions.Regex regexParseOrg = 
			new System.Text.RegularExpressions.Regex( 
				"^[ ]{3}(?<address>"+hex16+")[ \t]+(?<lineNumber>"+dec+")[ \t]+.org[ \t]+",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
			
		private static System.Text.RegularExpressions.Regex regexParseSymbolEqualsValue = 
			new System.Text.RegularExpressions.Regex( 
				"^[ ]{20}"+
				"(?<hex16>"+hex16+")"+
				"[ \t]+(?<lineNumber>"+dec+")"+
				"[ \t]+(?<symbol>"+symbol+")"+
				"[ \t]+=",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
			
		private static System.Text.RegularExpressions.Regex regexParseLabel = 
			new System.Text.RegularExpressions.Regex( 
				"^[ ]{3}(?<address>"+hex16+")[ \t]+(?<lineNumber>"+dec+")[ \t]+"+"(?<label>"+label+")(?<scope>"+scope+")$",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
		
		private static System.Text.RegularExpressions.Regex regexParseDS = 
			new System.Text.RegularExpressions.Regex( 
				"^[ ]{3}(?<address>"+hex16+")[ \t]+(?<lineNumber>"+dec+")[ \t]+.ds[ \t]+(?<count>"+hexOrDec+")$",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
		
		private static System.Text.RegularExpressions.Regex regexParseEmptyLine = 
			new System.Text.RegularExpressions.Regex( 
				"^[ ]{25}[ ]{0,5}"+
				"(?<lineNumber>"+dec+")$",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
		
		private static System.Text.RegularExpressions.Regex regexParseData = 
			new System.Text.RegularExpressions.Regex( 
				"^[ ]{3}"+
				"(?<address>"+hex16+")"+
				"[ ](?<instructionByte1>"+hex8+")"+
				"([ ](?<instructionByte2>"+hex8+"))?"+
				"([ ](?<instructionByte3>"+hex8+"))?"+
				"([ ](?<instructionByte4>"+hex8+"))?"+
				"([ ](?<instructionByte5>"+hex8+"))?"+
				"([ ](?<instructionByte6>"+hex8+"))?"+
				"[ \t]+(?<lineNumber>"+dec+")",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
		
		private static System.Text.RegularExpressions.Regex regexParseContinuedData = 
			new System.Text.RegularExpressions.Regex( 
				"^[ ]{8}"+
				"(?<instructionByte1>"+hex8+")"+
				"([ ](?<instructionByte2>"+hex8+"))?"+
				"([ ](?<instructionByte3>"+hex8+"))?"+
				"([ ](?<instructionByte4>"+hex8+"))?"+
				"([ ](?<instructionByte5>"+hex8+"))?"+
				"([ ](?<instructionByte6>"+hex8+"))?"+
				"$",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
		
		private static System.Text.RegularExpressions.Regex regexHexToValue = 
			new System.Text.RegularExpressions.Regex( 
				"^(0x|\\$)(?<digits>"+hex+"+)$",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
		
		private static System.Text.RegularExpressions.Regex regexDecToValue = 
			new System.Text.RegularExpressions.Regex( 
				"^"+dec+"$",
				System.Text.RegularExpressions.RegexOptions.Compiled 
			);
			

		/// <summary>
		/// Scan the hex file itself, and verify the stored data 
		/// is correct.  Also, pick up any additional bytes which are present, 
		/// but do not have source files (must be from library routines).
		/// </summary>
		private static void ScanHexFile(String filename)
		{
			String[] lines = FileCache.File(filename);

			foreach(String line in lines)
			{
				// handle a .hex file line...
				int bytesOfData = (line.Length - (1 + 2 + 4 + 2 + 2)) / 2;
				String dataMatch = "";
				for(int i=0;i<bytesOfData;i++)
					dataMatch += "(?<data"+i.ToString()+">"+hex8+")";
				
				System.Text.RegularExpressions.Regex regEx = 
					new System.Text.RegularExpressions.Regex(
						"^:"+
						"(?<LoadRecLen>"+hex8+")"+
						"(?<Offset>"+hex16+")"+
						"(?<RecType>"+hex8+")"+
						dataMatch+
						"(?<ChkSum>"+hex8+")"+
						"$"
					);
				System.Text.RegularExpressions.Match match = regEx.Match(line);
				if(match.Success == true)
				{
					byte loadRecLen = byte.Parse(
						SlashToBackslash(match.Groups["LoadRecLen"].Value),
						System.Globalization.NumberStyles.HexNumber
					);
					ushort offset = ushort.Parse(
						SlashToBackslash(match.Groups["Offset"].Value),
						System.Globalization.NumberStyles.HexNumber
					);
					byte recType = byte.Parse(
						SlashToBackslash(match.Groups["RecType"].Value),
						System.Globalization.NumberStyles.HexNumber
					);
					byte checksum = byte.Parse(
						SlashToBackslash(match.Groups["ChkSum"].Value),
						System.Globalization.NumberStyles.HexNumber
					);
					byte[] data = new byte[bytesOfData];
					for(int i=0;i<bytesOfData;i++)
					{
						data[i] = byte.Parse(
							SlashToBackslash(match.Groups["data"+i.ToString()].Value),
							System.Globalization.NumberStyles.HexNumber
						);
					}
					switch(recType)
					{
						case 0x00:	HexFile_HandleDataRecord(offset, data);			break;
						case 0x01:	HexFile_HandleEndOfFileRecord();				break;
						case 0x02:	HexFile_HandleExtendedSegmentRecord(data);		break;
						case 0x03:	HexFile_HandleStartSegmentAddressRecord();		break;
						case 0x04:	HexFile_HandleExtendedLinearAddressRecord();	break;
						case 0x05:	HexFile_HandleStartLinearAddressRecord();		break;
					}
				}
				else
				{
					System.Diagnostics.Debugger.Log(0, "HEXFile", "Unrparsed line: ["+line+"]\n");
				}
			}
		}
	
		/// <summary>
		/// Remember which Segment we're in, as we load a .hex file.
		/// </summary>
		private static bool hexReaderCurrentSegment = false;
		
		/// <summary>
		/// Read a Data Record
		/// </summary>
		private static void HexFile_HandleDataRecord(ushort offset, byte[] data)
		{
			for(int i=0;i<data.Length;i++)
			{
				FlashByte flashByte = code[offset+i+(hexReaderCurrentSegment?0x10000:0)];
				if(flashByte.IsBurned)
				{
					// verify that it was burned correctly...
					System.Diagnostics.Debug.Assert(flashByte.Value == data[i]);
				}
				else
				{
					// burn some code which must have come from a library...
					flashByte.Burn(data[i], null, 0);
				}
			}
		}
		
		/// <summary>
		/// Read an EOF Record.
		/// </summary>
		private static void HexFile_HandleEndOfFileRecord()
		{
			// do nothing.
		}
		
		/// <summary>
		/// Read an Extended Segment Record.
		/// </summary>
		private static void HexFile_HandleExtendedSegmentRecord(byte[] data)
		{
			System.Diagnostics.Debug.Assert( (data[0] == 0) || (data[0] == 0x10) );
			System.Diagnostics.Debug.Assert(data[1] == 0);
			hexReaderCurrentSegment = ( (data[0] != 0) || (data[1] != 0) ) ? true : false;
		}
		
		/// <summary>
		/// Not Implemented.
		/// </summary>
		private static void HexFile_HandleStartSegmentAddressRecord()
		{
			System.Diagnostics.Debug.Assert(false);
		}
		
		/// <summary>
		/// Not Implemented.
		/// </summary>
		private static void HexFile_HandleExtendedLinearAddressRecord()
		{
			System.Diagnostics.Debug.Assert(false);
		}
		
		/// <summary>
		/// Not Implemented.
		/// </summary>
		private static void HexFile_HandleStartLinearAddressRecord()
		{
			System.Diagnostics.Debug.Assert(false);
		}
		
		/// <summary>
		/// Convert '/'s to '\'s
		/// </summary>
		private static String SlashToBackslash(String text)
		{
			String t = "";
			foreach(char c in text)
			{
				if(c == '/')
					t = t + '\\';
				else
					t = t + c;
			}
			return t;
		}
		
		
		/// <summary>
		/// When a source file is clicked on under the View | Source File... submenu, 
		/// this event handler causes the Source View to repaint, showing the desired file.
		/// </summary>
		private static void ViewSourceFile_Click(object sender, System.EventArgs e)
		{   
			System.Windows.Forms.MenuItem theItem = (System.Windows.Forms.MenuItem)sender;

			String filename = SlashToBackslash(theItem.Text);
			MainForm.SourceViewJumpTo(filename);
		}
		
		
		/// <summary>
		/// search a .map file for a list of .rel files -> .rst files to scan.
		/// </summary>
		private static void ScanMapFile(System.Windows.Forms.MenuItem sourceFileMenuItem, String mapFileToScan)
		{
			System.Text.RegularExpressions.Match match = null;

			String[] lines = FileCache.File(mapFileToScan);

			// search for .rel (.rst) files to load!
			int lineNumber = 1;
			foreach(String line in lines)
			{
				// handle a .module line...
				match = regexScanMapFileModule.Match(line);
				if(match.Success == true)
				{
					// figure out what the file/path is.
					currentFilename = SlashToBackslash(match.Groups["file"].Value+".rst");
					AddFileToSourceMenu(currentFilename);
					
					// scan the file for code/data.
					System.Diagnostics.Debugger.Log(0, "ScanRstFile", "Scanning file "+currentFilename+"...\n");
					ScanRstFile(
						FileCache.TheCachedFile(currentFilename)
					);
				}
				lineNumber++;
			}

			// search for global variables to display in the Globals Form!
			lineNumber = 1;
			String segName = null;		// name of current segment
			ushort segStartAddress = 0;	// location of current segment
			ushort segSize = 0;			// number of bytes in current segment
			ushort symbolAddress = 0;	// address of the last symbol seen
			String symbolName = null;	// name of the last symbol seen
			int pre = 0;				// index of regex with correct pre-word, on all entries of this segment.
			foreach(String line in lines)
			{
				// handle an XSEG line...
				// XSEG  06CD   0850 =   2128. bytes (REL,CON,XDATA)
				match = regexScanMapFileXSEG.Match(line);
				if(match.Success == true)
				{
					// commit last symbol seen
					if(symbolName != null)
						GlobalsForm.AddSymbol(segName, symbolName, symbolAddress, (ushort)(segStartAddress + segSize - symbolAddress));

					// remember the size of XSEG.
					segName = SlashToBackslash(match.Groups["segment"].Value);
					segStartAddress = ushort.Parse(SlashToBackslash(match.Groups["startAddress"].Value), System.Globalization.NumberStyles.HexNumber);
					segSize = ushort.Parse(SlashToBackslash(match.Groups["size"].Value), System.Globalization.NumberStyles.HexNumber);
					if(String.Compare(segName,"XSEG", true) == 0)
					{
						pre = 3;
					}
					else if(String.Compare(segName,"DSEG", true) == 0)
					{
						pre = 2;
					}
					else if(String.Compare(segName,"BSEG", true) == 0)
					{
						pre = 0;
					}
					else if(String.Compare(segName,"CSEG", true) == 0)
					{
						pre = 1;
					}
					symbolName = null;	// ensure 
				}
				else if(segName != null)
				{
					// try to match a symbol line...
					// 0D:06CD  __mbrTask
					match = regexScanMapFileSymbol[pre].Match(line);
					if(match.Success == true)
					{
						ushort newSymbolAddress = ushort.Parse(SlashToBackslash(match.Groups["addr"].Value), System.Globalization.NumberStyles.HexNumber);
						String newSymbolName = SlashToBackslash(match.Groups["symbol"].Value);
						
						// commit previous symbol seen, now that we know how large it is 
						// (because of where the new symbol starts)
						if(symbolName != null)
							GlobalsForm.AddSymbol(segName, symbolName, symbolAddress, (ushort)(newSymbolAddress - symbolAddress));

						// remember new symbol we're collecting.
						symbolAddress = newSymbolAddress;
						symbolName = newSymbolName;
					}	
				}
				else 
				{
					// try to match the end of a segment...
					// Hexadecimal
					match = regexScanMapFileHexadecimal.Match(line);
					if(match.Success == true)
					{
						segName = null;	// not in a segment!
					}	
				}
				lineNumber++;
			}
		}
		
		
		/// <summary>
		/// Adds the name of a file to the View | Source File... menu
		/// </summary>
		private static void AddFileToSourceMenu(String fn)
		{
			// add the file to the list of browsable source.
			// TODO:  Only add this file to the view menu if it's not already present!
			fn = SlashToBackslash(fn);
			foreach(System.Windows.Forms.MenuItem menuItem in MainForm.SourceFileMenu.MenuItems)
				if(String.Compare(menuItem.Text, fn) == 0)
					return;	// found it already in the list!!!
					
			// add it to the list.
			System.Windows.Forms.MenuItem item = 
				new System.Windows.Forms.MenuItem(fn);
			item.Click += new System.EventHandler(ViewSourceFile_Click);
			MainForm.SourceFileMenu.MenuItems.Add(item);
		}
		

		/// <summary>
		/// Process the lines of a .rst file
		/// </summary>
		private static void ScanRstFile(CachedFile theCachedFile)
		{
			int lineNumber = 1;
			hasCurrentAddress = false;	// start out not knowing what the current address is...
			for(int n=0;n<theCachedFile.Lines.Length;n++)
				ParseRstLine(theCachedFile, lineNumber++);
		}
		

		/// <summary>
		/// Parse a line of text from a .rst file
		/// </summary>
		private static void ParseRstLine(CachedFile theCachedFile, int lineNumber)
		{
//			System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Parse "+lineNumber.ToString()+" ["+rawLine+"]\n");
			currentLineNumber = lineNumber;
			currentCachedFile = theCachedFile;
			
			String line = RstTrimLine(theCachedFile.Lines[lineNumber-1]);
			
			if(RstTryParseModule(line))				return;
			if(RstTryParseGlobl(line))				return;
			if(RstTryParseArea(line))				return;
			if(RstTryParseOrg(line))				return;
			if(RstTryParseSymbolEqualsValue(line))	return;
			if(RstTryParseLabel(line))				return;
			if(RstTryParseDS(line))					return;
			if(RstTryParseData(line))				return;
			if(RstTryParseContinuedData(line))		return;
			if(RstTryParseEmptyLine(line))			return;
			
			System.Diagnostics.Debugger.Log(0, "ParseRstLine", "No idea how to parse ["+lineNumber+"]!\n");
			System.Diagnostics.Debug.Assert(false);
		}

		/// <summary>
		/// Parse a line such as '6  .module blah'
		/// </summary>
		private static bool RstTryParseModule(String line)
		{
			// handle a .module line...
			System.Text.RegularExpressions.Match match = regexParseModule.Match(line);
			if(match.Success == true)
			{
				// we don't really care about these!
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "RstLine# = ["+match.Groups["lineNumber"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Module = ["+match.Groups["module"].Value+"]\n");
				return true;
			}
			return false;
		}
		
		/// <summary>
		/// Returns the address in the flash which is associated with a specific file & line-number.
		/// </summary>
		public static int FindAddress(String fn, int lineNumber)
		{
			object addr = FileCache.TheCachedFile(fn).AddressOfLine(lineNumber);
			if(addr != null)
				return ((ushort)addr) + (FileCache.TheCachedFile(fn).SegmentOfLine(lineNumber)?0x10000:0);

			// if there's no executable code on this line, try the next line (if there is one!)
			if(lineNumber < FileCache.File(fn).Length)
				return FindAddress(fn, lineNumber+1);
			else
				return 0x0000;	// off bottom of file!
		}
		
		/// <summary>
		/// Parse a line such as '6  .module blah'
		/// </summary>
		private static bool RstTryParseGlobl(String line)
		{
			// handle a .module line...
			System.Text.RegularExpressions.Match match = regexParseGlobl.Match(line);
			if(match.Success == true)
			{
				// we don't really care about these!
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "RstLine# = ["+match.Groups["lineNumber"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Symbol = ["+match.Groups["symbol"].Value+"]\n");
				return true;
			}
			return false;
		}
		
		
		/// <summary>
		/// Parse a line such as '6  .area blah (blah)'
		/// </summary>
		private static bool RstTryParseArea(String line)
		{
			// handle a .area line...
			System.Text.RegularExpressions.Match match = regexParseArea.Match(line);
			if(match.Success == true)
			{
				// we don't really care about these!
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "RstLine# = ["+match.Groups["lineNumber"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "SegmentName = ["+match.Groups["segmentName"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "SegmentModes = ["+match.Groups["segmentModes"].Value+"]\n");
				return true;
			}
			return false;
		}
		
		
		/// <summary>
		/// Parse a line such as '6  .org blah'
		/// </summary>
		private static bool RstTryParseOrg(String line)
		{
			// handle a .org line...
			System.Text.RegularExpressions.Match match = regexParseOrg.Match(line);
			if(match.Success == true)
			{
				// we don't really care about these!
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "RstLine# = ["+match.Groups["lineNumber"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Address = ["+match.Groups["address"].Value+"]\n");
				return true;
			}
			return false;
		}
		
		
		/// <summary>
		/// Parse a line such as '4000   6  someSymbol = 0x4000'
		/// </summary>
		private static bool RstTryParseSymbolEqualsValue(String line)
		{
			// handle a symbol assignment...
			System.Text.RegularExpressions.Match match = regexParseSymbolEqualsValue.Match(line);
			if(match.Success == true)
			{
				// we don't really care about these!
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "RstLine# = ["+match.Groups["lineNumber"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Symbol = ["+match.Groups["symbol"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Value = ["+match.Groups["hex16"].Value+"]\n");
				return true;
			}
			return false;
		}
		
		
		/// <summary>
		/// Parse a line such as '6  someSymbol::'
		/// </summary>
		private static bool RstTryParseLabel(String line)
		{
			// handle a label...
			System.Text.RegularExpressions.Match match = regexParseLabel.Match(line);
			if(match.Success == true)
			{
				// we don't really care about these!
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "RstLine# = ["+match.Groups["lineNumber"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Address = ["+match.Groups["address"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Label = ["+match.Groups["label"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Scope = ["+match.Groups["scope"].Value+"]\n");
				return true;
			}
			return false;
		}
		

		/// <summary>
		/// Parse a line such as '2e00   6  .ds 1'
		/// </summary>
		private static bool RstTryParseDS(String line)
		{
			// handle a storage line...
			System.Text.RegularExpressions.Match match = regexParseDS.Match(line);
			if(match.Success == true)
			{
				// we don't really care about these!
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "RstLine# = ["+match.Groups["lineNumber"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Address = ["+match.Groups["address"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Count = ["+match.Groups["count"].Value+"]\n");
				return true;
			}
			return false;
		}

		
		/// <summary>
		/// Parse a line such as '6 ; comments...'
		/// </summary>
		private static bool RstTryParseEmptyLine(String line)
		{
			// handle a line of comments...
			System.Text.RegularExpressions.Match match = regexParseEmptyLine.Match(line);
			if(match.Success == true)
			{
				// we don't really care about these!
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "RstLine# = ["+match.Groups["lineNumber"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Comment = ["+match.Groups["comment"].Value+"]\n");
				return true;
			}
			return false;
		}
		
		
		/// <summary>
		/// Parse a line such as '4000 20 30 40   6  ljmp 34$'
		/// </summary>
		private static bool RstTryParseData(String line)
		{
			// handle a line containing an instruction...
			System.Text.RegularExpressions.Match match = regexParseData.Match(line);
			if(match.Success == true)
			{
				// we don't really care about these!
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "RstLine# = ["+match.Groups["lineNumber"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Address = ["+match.Groups["address"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte1"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte2"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte3"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte4"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte5"].Value+"]\n");
//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte6"].Value+"]\n");
				CurrentAddress = ToValue("$"+match.Groups["address"].Value);
				
				currentCachedFile.AssignAddressOfLine(currentLineNumber, (ushort)CurrentAddress, A16);
				
				if(match.Groups["instructionByte1"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte1"].Value));
				if(match.Groups["instructionByte2"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte2"].Value));
				if(match.Groups["instructionByte3"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte3"].Value));
				if(match.Groups["instructionByte4"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte4"].Value));
				if(match.Groups["instructionByte5"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte5"].Value));
				if(match.Groups["instructionByte6"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte6"].Value));
				return true;
			}
			return false;
		}
		

		/// <summary>
		/// Parse a line such as '     20 30 40   6'
		/// </summary>
		private static bool RstTryParseContinuedData(String line)
		{
			// handle a line containing an instruction...
			System.Text.RegularExpressions.Match match = regexParseContinuedData.Match(line);
			if(match.Success == true)
			{
				// we don't really care about these!
				//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte1"].Value+"]\n");
				//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte2"].Value+"]\n");
				//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte3"].Value+"]\n");
				//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte4"].Value+"]\n");
				//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte5"].Value+"]\n");
				//				System.Diagnostics.Debugger.Log(0, "ParseRstLine", "Instruction = ["+match.Groups["instructionByte6"].Value+"]\n");

				currentCachedFile.AssignAddressOfLine(currentLineNumber, (ushort)CurrentAddress, A16);

				if(match.Groups["instructionByte1"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte1"].Value));
				if(match.Groups["instructionByte2"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte2"].Value));
				if(match.Groups["instructionByte3"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte3"].Value));
				if(match.Groups["instructionByte4"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte4"].Value));
				if(match.Groups["instructionByte5"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte5"].Value));
				if(match.Groups["instructionByte6"].Success)
					BurnToCurrentAddress((byte)ToValue("$"+match.Groups["instructionByte6"].Value));
				return true;
			}
			return false;
		}
		
		
		/// <summary>
		/// Convert a decimal or hexadecimal value into an integral value.
		/// </summary>
		private static int ToValue(String val)
		{
			int v = 0;

			// try a hex number
			System.Text.RegularExpressions.Match match = regexHexToValue.Match(val);
			if(match.Success == true)
			{
				foreach(char digit in match.Groups["digits"].Value.ToUpper())
				{
					if( (digit >= '0') && (digit <= '9') )
					{
						v = v * 16 + (int)(digit-'0');
					}
					else if( (digit >= 'A') && (digit <= 'F') )
					{
						v = v * 16 + (int)(digit-'A'+10);
					}
					else
					{
						System.Diagnostics.Debug.Assert(false);	// lost!
					}
				}
			}
			else
			{
				// try a decimal number
				match = regexDecToValue.Match(val);
				if(match.Success == true)
				{
					foreach(char digit in match.Groups["digits"].Value.ToUpper())
					{
						if( (digit >= '0') && (digit <= '9') )
						{
							v = v * 10 + (int)(digit-'0');
						}
					}
				}
				else
				{
					System.Diagnostics.Debug.Assert(false);	// can't parse the number
				}
			}
			return v;
		}
		
		
		/// <summary>
		/// CurrentAddress is used internally, while burning code/data into the flash.
		/// </summary>
		private static int currentAddress;	// 0x0000..0xffff   -- use A16 to select segment!

		/// <summary>
		/// Remembers whether we have GOT a starting address before we begin burning successive bytes.
		/// </summary>
		private static bool hasCurrentAddress;

		/// <summary>
		/// Remembers the current address in flash, where we are burning a series of bytes.
		/// </summary>
		private static int CurrentAddress
		{
			get
			{
				System.Diagnostics.Debug.Assert(hasCurrentAddress);
				return currentAddress;
			}
			set
			{
				currentAddress = value;
				hasCurrentAddress = true;
			}
		}
		
		/// <summary>
		/// Remembers the current file that this data is associated with
		/// </summary>
		private static String currentFilename;
		
		/// <summary>
		/// Remembers the current line number where this data is associated with
		/// </summary>
		private static int currentLineNumber;
		
		private static CachedFile currentCachedFile = null;
		
		/// <summary>
		/// Burns a byte into the Flash at the current address.  Advances the current address by one byte.
		/// </summary>
		private static void BurnToCurrentAddress(byte v)
		{
			Burn(CurrentAddress, v, currentFilename, currentLineNumber);
		}
		
		
		/// <summary>
		/// Burn a byte into the Flash.  Assigns a new "CurrentAddress", File, and line.
		/// </summary>
		private static void Burn(int address, byte v, String fn, int line)
		{
			currentFilename = fn;
			currentLineNumber = line;
			CurrentAddress = address;

			code[CurrentAddress + ((A16==true) ? 0x10000 : 0x00000)].Burn(v, fn, line);
			
			CurrentAddress++;
			if(CurrentAddress == 0x10000)
				CurrentAddress = 0x0000;
		}
		
		
		/// <summary>
		/// Remove comments from the line of text being parsed
		/// </summary>
		private static String RstTrimLine(String line)
		{
			// remove excess white-space.
			line = line.TrimEnd();
			for(int n=0;n<line.Length;n++)
			{
				if(line[n] == ';')
				{
					line = line.Substring(0, n).TrimEnd();
					break;
				}
			}
			return line;
		}

		
		/// <summary>
		/// The current state of A16, the bank selection pin.
		/// </summary>
		public static bool a16;	// current state of pin A16 (segment select)
		
		/// <summary>
		/// The current state of A16, the bank selection pin.
		/// </summary>
		public static bool A16
		{
			get
			{
				return a16;
			}
			set
			{
				a16 = value;
			}
		}
		
		
		/// <summary>
		/// Discover the source file which was the source of this byte.
		/// 0x0000..0xffff
		/// </summary>
		public static String RstSourceFile(ushort flashOffset, bool a16)
		{
			return code[flashOffset + ((a16==true) ? 0x10000 : 0x00000)].SourceFile;
		}
		
		
		/// <summary>
		/// Discover the line number in a source file which was the source of this byte.
		/// </summary>
		public static int RstSourceFileLineNumber(ushort flashOffset, bool a16)
		{
			return code[flashOffset + ((a16==true) ? 0x10000 : 0x00000)].SourceFileLineNumber;
		}

		/// <summary>
		/// Get a value at a specific address in the flash ram.
		/// Uses the 16-bit addressing with A16 flag.
		/// </summary>
		public static byte GetByte(ushort address, bool a16)
		{
			return code[address+(a16?0x10000:0)].Value;
		}

		/// <summary>
		/// Get a value at a specific address in the flash ram.
		/// Uses the 16-bit addressing with A16 flag.
		/// </summary>
		public static ushort GetWord(ushort address, bool a16)
		{
			return (ushort)(
				code[address+(a16?0x10000:0)+0].Value + 
				code[address+(a16?0x10000:0)+1].Value * 256);
		}

		/// <summary>
		/// Get a value at a specific address in the flash ram.
		/// Uses the 16-bit addressing with A16 flag.
		/// </summary>
		public static ulong GetLong(ushort address, bool a16)
		{
			return 
				( (ulong)code[address+(a16?0x10000:0)+0].Value) + 
				(((ulong)code[address+(a16?0x10000:0)+1].Value)<<8) +
				(((ulong)code[address+(a16?0x10000:0)+1].Value)<<16) +
				(((ulong)code[address+(a16?0x10000:0)+1].Value)<<24);
		}

		/// <summary>
		/// Get a value at a specific address in the flash ram.
		/// Uses linear addressing on the Flash Chip.
		/// </summary>
		public static byte GetByteLinear(int rawFlashOffset)
		{
			return code[rawFlashOffset].Value;
		}

		/// <summary>
		/// A new breakpoint is added to flash-memory.
		/// </summary>
		public static void AddBreakpoint(Breakpoint b)
		{
			breakpoints[b.Address].Add(b);
		}

		/// <summary>
		/// Remove a breakpoint which was in flash-memory.
		/// </summary>
		public static void DestroyBreakpoint(Breakpoint b)
		{
			breakpoints[b.Address].Remove(b);
		}
		
		/// <summary>
		/// Returns a breakpoint from the desired address, if one exists, or NULL.
		/// NOTE:  The Breakpoint may or may not be active!!!
		/// </summary>
		public static Breakpoint BreakpointAt(int addr)
		{
			System.Diagnostics.Debug.Assert((addr>=0) && (addr <= 0x1ffff));
			if(breakpoints[addr].Count > 0)
				foreach(Breakpoint b in breakpoints[addr])
					return b;
			return null;
		}
		
		/// <summary>
		/// Returns an *ACTIVE* breakpoint from the desired address, if one exists, or NULL.
		/// </summary>
		public static Breakpoint ActiveBreakpointAt(int address)
		{
			System.Diagnostics.Debug.Assert((address>=0) && (address <= 0x1ffff));
			if(breakpoints[address].Count > 0)
				foreach(Breakpoint b in breakpoints[address])
					if(b.IsActive)
						return b;
			return null;
		}
		
		/// <summary>
		/// Enough ranm to hold all the bytes which can be burned into the flash chip.
		/// this array will be populated by the Scan...() functions.
		/// </summary>
		private static FlashByte[] code = new FlashByte[65536*2];

	
		/// <summary>
		/// A collection of breakpoints which are watching each CPU address.
		/// </summary>
		private static System.Collections.ArrayList[] breakpoints = new System.Collections.ArrayList[65536*2];	// each entry is an arraylist of breakpoints.
	}
}
