using System;

namespace _8052Sim
{
	/// <summary>
	/// IdeDevice is a Harddrive, connected to the IDE cable.
	/// NOTE: in the future, should we desire to emulate CDROM's, etc. 
	/// an dinterface should be substituted here.  But at this point, 
	/// I have no idea what it would look like.
	/// </summary>
	public class IdeDevice
	{
		public enum ErrorRegisterDiagnosticCode
		{
			NoErrorDetected = 0x01,
			FormatterDeviceError = 0x02,
			SectorBufferError = 0x03,
			EccSecurityError = 0x04,
			ControllerMicroprocessorError = 0x05,
			Device1Failed = 0x80,
		}
		
		public enum Command
		{
			NoCommand					= 0x00,
			InitializeDriveParameters	= 0x91,
			SetMultiple					= 0xC6,
			ReadMultiple				= 0xC4,
			StandBy						= 0xE0,
			IdentifyDevice				= 0xEC,
		}

		/// <summary>
		/// Create and configure the device.
		/// </summary>
		public IdeDevice(bool _isDevice0, String filename)
		{
			isDevice0 = _isDevice0;
			isPresent = (filename != "");
			deviceHeadRegister = 0xa0;
//			errorRegister = NoErrorDetected;
			ideFeatureRegister = 0x00;
			sectorCountRegister = 0x01;
			sectorNumberRegister = 0x01;
			cylinderLowRegister = 0x00;
			cylinderHighRegister = 0x00;
//			statusRegister = 0x50;
			alternateStatusRegister = 0x50;
			commandRegister = Command.NoCommand;
			
			readMultipleBlockSize = 0;
			
			readSectorQueue = new System.Collections.ArrayList();
			SetupDriveIdentificationData();
			
			if(isPresent)
			{
				reader = System.IO.File.OpenRead("C:\\MP3PlayerHardware\\Source\\Drive Images\\"+filename);
			}
		}

		/// <summary>
		/// Set the device/head register.  As a small optimization, the Ide code
		/// remembers the last write to this register, and issues all read/write 
		/// operations only to the correct device, so that the device code itself 
		/// does not need to watch the Device flag in this register.
		/// </summary>
		public byte DeviceHeadRegister
		{
			set	
			{
				deviceHeadRegister = value;
			}
		}
		
		/// <summary>
		/// Get the Alternate Status Register value.
		/// </summary>
		public byte AlternateStatusRegister
		{
			get
			{
				return alternateStatusRegister;			
			}
		}
		
		/// <summary>
		/// Set the Sector Number Register.
		/// </summary>
		public byte SectorNumberRegister
		{
			set
			{
				sectorNumberRegister = value;
			}
		}
		
		public byte CylinderLowRegister
		{
			set
			{
				cylinderLowRegister = value;
			}
		}
	
		public byte CylinderHighRegister
		{
			set
			{
				cylinderHighRegister = value;
			}
		}
	
		public byte SectorCountRegister
		{
			set
			{
				sectorCountRegister = value;
			}
		}
	
		public byte IdeFeatureRegister
		{
			set
			{
				ideFeatureRegister = value;
			}
		}
		
		public byte CommandRegister
		{
			set
			{
				commandRegister = (Command)value;
				
				if(isPresent)
				{
					// begin executing the command...
					switch(commandRegister)
					{
						case Command.NoCommand:
							break;

						case Command.StandBy:
							// do nothing...
							break;

						case Command.IdentifyDevice:
							readSectorQueue.Clear();
							readSectorQueue.Add(identificationSectorFujitsu);
							alternateStatusRegister |= 0x08;	// notify CPU some data is ready to be sent.
							pioOffset = 0x100;					// ready to send a new sector...
							break;
						
						case Command.InitializeDriveParameters:
							sectorsPerTrack = sectorCountRegister;
							headsPerCylinder = (byte)((deviceHeadRegister & 0x0f)+1);
							alternateStatusRegister |= 0x08;
							break;
						
						case Command.SetMultiple:
							switch(sectorCountRegister)
							{
								case 0:
								case 2:
								case 4:
								case 8:
								case 16:
									readMultipleBlockSize = sectorCountRegister;
									break;
									
								default:
									System.Diagnostics.Debug.Assert(false);	// invalid setting!!!
									break;
							}
							alternateStatusRegister |= 0x08;
							break;
						
						case Command.ReadMultiple:
							{
								int sectors = sectorCountRegister;
								if(sectors == 0)
									sectors = 256;
								if( (deviceHeadRegister & 0x40) != 0)
								{
									ulong lbaAddress = 
										((ulong)sectorNumberRegister) | 
										(((ulong)cylinderLowRegister)<<8) |
										(((ulong)cylinderHighRegister)<<16) |
										((((ulong)deviceHeadRegister)&0x0f)<<24);
									LBA_ReadMultiple(sectors, lbaAddress);
								}
								else
								{
//									CHS_ReadMultiple(sectors);
									System.Diagnostics.Debug.Assert(false);
								}
							}
							break;
						
						default:
							System.Diagnostics.Debug.Assert(false);
							break;
					}
				}
			}
		}
		

		/// <summary>
		/// Read a number of sectors from a specific LBA address on the drive.
		/// </summary>
		private void LBA_ReadMultiple(int sectors, ulong lbaAddress)
		{
			long fileOffset = (long)(lbaAddress * 512);
			readSectorQueue.Clear();
			for(int n=0;n<readMultipleBlockSize;n++)
			{
				ushort[] sector = new ushort[256];
				byte[] sectorBytes = new byte[512];
				reader.Seek(fileOffset, System.IO.SeekOrigin.Begin);
				reader.Read(sectorBytes, 0, sectorBytes.Length);
				for(int i=0;i<sector.Length;i++)
					sector[i] = (ushort)(sectorBytes[i*2]+256*sectorBytes[i*2+1]);
				readSectorQueue.Add(sector);
				fileOffset += 512;
			}
		}


		/// <summary>
		/// The Programmed I/O port (PIO) interface.
		/// </summary>
		public ushort PIO
		{
			get
			{
				if(pioOffset == 0x100)	// need a new sector to send?
				{
					pioOffset = 0;
					if(readSectorQueue.Count>0)	// are there any more sectors?
					{
						pioSector = (ushort[])(readSectorQueue[0]);	// this is the sector we will send...
						readSectorQueue.RemoveAt(0);	// remove it from the queue
					}
					else
					{
//						System.Diagnostics.Debug.Assert(false);	// someone wanted more data than was available!!!
						System.Diagnostics.Debugger.Log(0, "IDE", "Ide - Warning: Device Read attempted for too many sectors!\n");
						pioSector = new ushort[256];			// send an empty sector?
					}
				}
				return pioSector[pioOffset++];	// send a short from the current sector.
			}
		}
		
		private ushort[] pioSector;	// the current sector being sent/recieved
		private int pioOffset;		// the offset within the sector of the next byte to send (set to 0x100 to have a new sector fetched)
		
		/// <summary>
		/// Last known value of this register.
		/// </summary>
		private byte deviceHeadRegister;
		
		/// <summary>
		/// Last known value of this register.
		/// </summary>
		private byte alternateStatusRegister;
		
		/// <summary>
		/// Last known value of this register.
		/// </summary>
		private byte sectorNumberRegister;
		
		/// <summary>
		/// Last known value of this register.
		/// </summary>
		private byte cylinderLowRegister;
		
		/// <summary>
		/// Last known value of this register.
		/// </summary>
		private byte cylinderHighRegister;
		
		/// <summary>
		/// Last known value of this register.
		/// </summary>
		private byte sectorCountRegister;
		
		/// <summary>
		/// Last known value of this register.
		/// </summary>
		private byte ideFeatureRegister;

		/// <summary>
		/// Last known value of this register.
		/// </summary>
		private IdeDevice.Command commandRegister;
		
		/// <summary>
		/// TRUE if it's device#0, FALSE if it's device#1.
		/// </summary>
		private bool isDevice0;

		/// <summary>
		/// TRUE if the device is actually present
		/// </summary>
		private bool isPresent;
		
		/// <summary>
		/// The sector of data which is returned on a Drive Identification Request.
		///	Manufact:	Fujitsu 
		///	Model:		MPG3409AT
		///	S/N:		VH06T1304M35
		/// </summary>
		private ushort[] identificationSectorFujitsu;

		/// <summary>
		/// The sector of data which is returned on a Drive Identification Request.
		///	Manufact:	IBM TravelStar 
		///	Model:		IC25N020ATDA04-0
		///	S/N:		63J62680
		/// </summary>
		private ushort[] identificationSectorTravelStar;

		/// <summary>
		/// Queue of sectors to send to DRAM.
		/// </summary>
		private System.Collections.ArrayList readSectorQueue;	// byte[512]'s (sectors) to send to DRAM.


		/// <summary>
		/// The current setting for Sectors/Track (0..255)
		/// </summary>
		private byte sectorsPerTrack;

		/// <summary>
		/// The current setting for Heads/Cylinder (1..16)
		/// </summary>
		private byte headsPerCylinder;
		
		/// <summary>
		/// The number of sectors to send on a read-multiple command.
		/// (0, 2, 4, 8, or 16)
		/// </summary>
		private byte readMultipleBlockSize;
	

		/// <summary>
		/// The drive image file we're reading from...
		/// </summary>
		System.IO.FileStream reader;
		
		#region DRIVE IDENTIFICATION STRUCTURES
		private void SetupDriveIdentificationData()
		{
			identificationSectorFujitsu = new ushort[] 
			{
				0x5A04, 0xFF3F, 0x0000, 0x1000, 0x0000, 0x0000, 0x3F00, 0x0000,
				0x0000, 0x0000, 0x2020, 0x2020, 0x2020, 0x2020, 0x4856, 0x3630,
				0x3154, 0x3033, 0x4D34, 0x3533, 0x0300, 0x0010, 0x0400, 0x3238,
				0x3942, 0x0000, 0x0000, 0x5546, 0x494A, 0x5354, 0x2055, 0x504D,
				0x3347, 0x3034, 0x4139, 0x2054, 0x4520, 0x2020, 0x2020, 0x2020,
				0x2020, 0x2020, 0x2020, 0x2020, 0x2020, 0x2020, 0x2020, 0x1080,
				0x0000, 0x002B, 0x0040, 0x0002, 0x0002, 0x0700, 0xFF3F, 0x1000,
				0x3F00, 0x10FC, 0xFB00, 0x0000, 0xC0AB, 0xC504, 0x0000, 0x0704,
				0x0300, 0x7800, 0x7800, 0xF000, 0x7800, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x3E00, 0x1500, 0x6B34, 0x0843, 0x0040, 0x6134, 0x0000, 0x0040,
				0x3F00, 0x1000, 0x0000, 0x0000, 0x0000, 0x006B, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0100, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
			};

			identificationSectorTravelStar = new ushort[]
			{
				0x5A04, 0xFF3F, 0x37C8, 0x1000, 0x0000, 0x0000, 0x3F00, 0x0000,
				0x0000, 0x0000, 0x2020, 0x2020, 0x2020, 0x2020, 0x3620, 0x4133,
				0x3336, 0x364A, 0x3632, 0x3038, 0x0300, 0x1C0E, 0x0400, 0x4144,
				0x4F33, 0x3741, 0x4136, 0x4349, 0x3532, 0x304E, 0x3032, 0x5441,
				0x4144, 0x3430, 0x302D, 0x2020, 0x2020, 0x2020, 0x2020, 0x2020,
				0x2020, 0x2020, 0x2020, 0x2020, 0x2020, 0x2020, 0x2020, 0x1080,
				0x0000, 0x000F, 0x0040, 0x0002, 0x0002, 0x0700, 0xFF3F, 0x1000,
				0x3F00, 0x10FC, 0xFB00, 0x0001, 0x8029, 0x5402, 0x0000, 0x0704,
				0x0300, 0x7800, 0x7800, 0xF000, 0x7800, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x3C00, 0x1300, 0x6B74, 0xA849, 0x0340, 0x68F4, 0x0808, 0x0340,
				0x3F00, 0x0C00, 0x0000, 0x8040, 0xFEFF, 0x0B60, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0100, 0x0B00, 0x0000, 0x0200, 0x0000, 0x0000, 0x0000, 0x0000,
				0xFC41, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x4F33, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
				0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xA5E4,
			};
		}
		#endregion
	}
}
