using System;

namespace _8052Sim
{
	/// <summary>
	/// Summary description for Sta013.
	/// </summary>
	public class Sta013
	{
		public static void Startup()
		{
			HardReset();
		}


		/// <summary>
		/// Ask the STA013 to perform a hard reset
		/// </summary>
		public static void HardReset()
		{
			wasClock = true;		// active high
			wasData = true;			// active high
			i2c_sendingByte = false;
			i2c_isReading = false;	
			
			i2c_ack = false;		// active low -> true = a LOW on the pin.
			i2c_sleeping = false;	// the state of this pin, on startup, should be irrelavent.

			SoftReset();
		}
		

		/// <summary>
		/// Ask the STA013 to perform a soft reset
		/// </summary>
		private static void SoftReset()
		{
			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 Soft Reset\n");
		}

		
		/// <summary>
		/// Discover the state of the I2C DATA pin, as known to the Sta013.
		/// i.e.  either the same data which was given to it, or FALSE
		/// if the Sta013 explicitly pulled it low.
		/// </summary>
		public static bool I2CAck(bool val)
		{
			// if the ACK signal is present in the STA013, then the I2C Data bus is pulled low by the STA013.
			if(i2c_ack == true)
				return false;
			
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - 8052 noticed SDA pin is "+val+"\n");

			return val;
		}
		
		
		/// <summary>
		/// Discover the state of the I2C CLOCK pin, as known to the Sta013.
		/// i.e.  either the same clock which was given to it, or FALSE
		/// if the Sta013 explicitly pulled it low.
		/// </summary>
		public static bool SclState()
		{
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - 8052 noticed SCL pin is "+wasClock+"\n");

			return wasClock;
		}
		
		
		/// <summary>
		/// Implements the CLK/DATA pin of the Sta013's I2C Interface.
		/// The state of these pins may be checked afterward, to see if the 
		/// STA013 has negated one of the pins.
		/// </summary>
		public static void I2CState(bool isClock, bool isData)
		{
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 sees SDA was "+wasData+", now "+isData+"\n");

			// START condition exists if a high-to-low transition 
			// occurs on the DATA pin while the clock pin is high.
			if( (wasClock == true) || (isClock == true) )
			{
				if(      (wasData == true) && (isData == false) )	I2C_Start();	// START condition!
				else if( (wasData == false) && (isData == true) )	I2C_Stop();		// STOP  condition!
			}
			
			if( (wasClock == false) && (isClock == true) )
			{
				if(i2c_isReading)
					I2C_SendBit();
				else
					I2C_RecieveBit(wasData);
			}
			else if( (wasClock == true) && (isClock == false) )
			{
				i2c_ack = false;	// assume no ACK...
				clockNumber++;
				if(clockNumber == 9)
				{
					I2C_CaptureDevAddr();
				}
				else if( ((clockNumber-10) % 9) == 8 )
				{
					if(!i2c_sleeping)
					{
						if(clockNumber == 18)
						{
							if(!i2c_sendingByte)
								if(!i2c_isReading)
									I2C_CaptureSubAddr();
						}
						else
						{
							I2C_CaptureData();
							i2c_ack = true;
						}
					}
				}
			}
			wasData = isData;
			wasClock = isClock;
		}
		
		
		/// <summary>
		/// I2C Bus experienced a START condition!
		/// </summary>
		private static void I2C_Start()
		{
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 Sees START Event\n");
			clockNumber = 0;
			bytesWritten.Clear();
		}


		/// <summary>
		/// I2C Bus recieved another bit.
		/// </summary>
		private static void I2C_RecieveBit(bool bit)
		{
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 Recieved a bit: "+bit+"\n");
			i2c_byte = (byte)(i2c_byte * 2 + (bit ? 1 : 0));
		}
		

		/// <summary>
		/// I2C Bus recieved another bit.
		/// </summary>
		private static void I2C_SendBit()
		{
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 Sent a bit: "+((i2c_byte & 0x80) == 0)+"\n");
			if(i2c_sendDelay)
				i2c_sendDelay = false;	// wait one clock before sending data, since ACK needs sent first!
			else
			{
				if((i2c_byte & 0x80) == 0)
					i2c_ack = true;	// remember - ACK = true means I2C Data line LOW until end of clock...
				else
					i2c_ack = false;
				i2c_byte *= 2;
			}
		}
		

		/// <summary>
		/// I2C Bus experienced a STOP condition!
		/// </summary>
		private static void I2C_Stop()
		{
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 sees a STOP\n");
//			if(clockNumber == 19)
			if(!i2c_isReading)
				I2C_ByteWrite();
			i2c_sendingByte = false;
			i2c_isReading = false;
		}


		/// <summary>
		/// This special number is the only device id that the STA013 will respond to.
		/// </summary>
		private static readonly byte STA013_UNIQUE_ID = 0x86;
		
		
		/// <summary>
		/// It is time to look at the i2c_byte keep it as a device address. 
		/// </summary>
		private static void I2C_CaptureDevAddr()
		{
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 captures DevAddr: "+i2c_byte+"\n");

			subAddressByte = i2c_byte;
			i2c_sleeping = true;
			if((i2c_byte & 0xfe) == STA013_UNIQUE_ID)
			{
				i2c_ack = true;
				i2c_sleeping = false;
				i2c_isReading = ((subAddressByte & 0x01) != 0) ? true : false;
				if(i2c_isReading)	// 8052 wishes to read a byte back from Sta013!
				{
					if(bytesNeedingRead.Count>0)
					{
						i2c_byte = (byte)(bytesNeedingRead[0]);	// place the byte into buffer, for transmation back to 8052!
						bytesNeedingRead.RemoveAt(0);
						i2c_sendDelay = true;	// but wait one cycle first, so ACK can be sent!
					}
				}
			}
		}


		/// <summary>
		/// It is time to look at the i2c_byte, and keep it as a device sub-address. 
		/// </summary>
		private static void I2C_CaptureSubAddr()
		{
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 captures a SubAddr: "+i2c_byte+"\n");
			subAddressByte = i2c_byte;
			i2c_ack = true;
			if(i2c_isReading)
			{
				I2C_ByteRead(subAddressByte);
				I2C_CaptureData();
				i2c_sendingByte = true;
			}
		}

		private static bool i2c_sendingByte;
		private static bool i2c_sendDelay;
		
		/// <summary>
		/// It is time to look at the i2c_byte, and apply it as data.
		/// or alternatively, if this is a write procedure, to place a new
		/// value in i2c_byte.
		/// </summary>
		private static void I2C_CaptureData()
		{
			if(i2c_isReading)
			{
				// reading data
				i2c_byte = (byte)(bytesNeedingRead[0]);
				bytesNeedingRead.RemoveAt(0);
			}
			else
			{
				// writing data
				bytesWritten.Add(i2c_byte);
			}
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 captures Data: "+i2c_byte+"\n");
		}
		

		/// <summary>
		/// This container holds all bytes which were part of the write operation, 
		/// where data is being written to STA013 from 8052.
		/// </summary>
		private static System.Collections.ArrayList bytesWritten = new System.Collections.ArrayList();	// bytes


		/// <summary>
		/// This container holds all bytes which are to be part of the read operation,
		/// where data is being read from STA013 and being sent back to 8052.
		/// </summary>
		private static System.Collections.ArrayList bytesNeedingRead = new System.Collections.ArrayList();	// bytes


		/// <summary>
		/// The i2c_byte contains the data to be written to STA013
		/// </summary>
		private static void I2C_ByteWrite()
		{
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 ByteWrite Operation\n");

			switch(subAddressByte)
			{
				case 0x01:	// Ident
					Ident();
					break;
				
				case 0x05:	// PLLCTL 7:0
				case 0x06:	// PLLCTL 20:16
				case 0x07:	// PLLCTL 15:12
					// do nothing
					break;
				
				case 0x08:	// (unknown)
				case 0x09:	// (unknown)
					// do nothing
					break;
				
				case 0x0b:	// (reserved)
					// do nothing
					break;
				
				case 0x0c:	// REQ_POL
					// do nothing
					break;
				
				case 0x0d:	// SCLK_POL
					// do nothing
					break;
				
				case 0x10:	// Soft Reset
					if((byte)(bytesWritten[0]) == 1)
						SoftReset();
					break;

				case 0x13:	// Play
					// TODO
//					System.Diagnostics.Debug.Assert(false);
					break;
					
				case 0x14:	// Mute
					// TODO
//					System.Diagnostics.Debug.Assert(false);
					break;
					
				case 0x18:	// DATA_REQ_ENABLE
					// do nothing
					break;
				
				case 0x20:	// (unknown)
				case 0x21:	// (unknown)
				case 0x22:	// (unknown)
				case 0x23:	// (unknown)
				case 0x24:	// (unknown)
				case 0x25:	// (unknown)
				case 0x26:	// (unknown)
				case 0x27:	// (unknown)
				case 0x28:	// (unknown)
				case 0x29:	// (unknown)
				case 0x2a:	// (unknown)
				case 0x3a:	// (unknown)
					// do nothing - these addresses are written to 
					// by initialization data file.
					break;
				
				case 0x50:	// MFSDF_441
				case 0x51:	// PLLFRAC_441_L
				case 0x52:	// PLLFRAC_441_H
				case 0x54:	// PCM DIVIDER
				case 0x55:	// PCMCONF
				case 0x61:	// MFSDF (X)
				case 0x64:	// PLLFRAC_L
				case 0x65:	// PLLFRAC_H
					// do nothing
					break;
				
				case 0x72:	// Run
					// do nothing
					break;
					
				case 0x77:	// TREBLE_FREQUENCY_LOW
				case 0x78:	// TREBLE_FREQUENCY_HIGH
				case 0x79:	// BASS_FREQUENCY_LOW
				case 0x7a:	// BASS_FREQUENCY_HIGH
					// do nothing
					break;
				
				default:
					System.Diagnostics.Debug.Assert(false);
					break;
			}
		}
		
		
		private static void Ident()
		{
			bytesNeedingRead.Clear();
			bytesNeedingRead.Add((byte)0xAC);
		}
		

		/// <summary>
		/// The i2c_byte contains the data to be sent to 8052.
		/// </summary>
		private static void I2C_ByteRead(byte subAddressByte)
		{
//			System.Diagnostics.Debugger.Log(0, "I2C", "I2C - STA013 ByteRead Operation\n");

			switch(subAddressByte)
			{
				default:
					System.Diagnostics.Debug.Assert(false);
					break;
			}
		}
		

		/// <summary>
		/// Last known state of the clock pin.
		/// </summary>
		private static bool wasClock;


		/// <summary>
		/// Last known state of the data pin.
		/// </summary>
		private static bool wasData;
		

		/// <summary>
		/// This is the byte is on the I2C bus. It might 
		/// be getting written to STA013, or might be getting 
		/// read from STA013!
		/// it is shifted left as each bit is dealt with, 
		/// until all 8 bits are done.
		/// </summary>
		private static byte i2c_byte;
		
		
		/// <summary>
		/// This byte is captured as it goes by in the i2c datastream.
		/// </summary>
		private static byte subAddressByte;


		/// <summary>
		/// This is the I2C clock number which we are currently on...
		/// 0..9
		/// </summary>
		private static byte clockNumber;
		

		/// <summary>
		/// This signal is reflected on the I2C bus by 
		/// being LOW durring the 9th clock.
		/// </summary>
		private static bool i2c_ack;	// Active LOW: a TRUE here means the pin sees a LOW!
		
		
		/// <summary>
		/// The device will sleep through any instructions which 
		/// were addressed to another device.
		/// </summary>
		private static bool i2c_sleeping;
		

		/// <summary>
		/// Whether this command wants to read or write to an internal
		/// register.
		/// </summary>
		private static bool i2c_isReading;
	}
}
