using System;

namespace _8052Sim
{
	/// <summary>
	/// Summary description for Xilinx.
	/// </summary>
	public class Xilinx
	{
		/// <summary>
		/// Summary description for Xilinx.
		/// </summary>
		public static void Startup()
		{
			writer = System.IO.File.Open("simulation.mp3", System.IO.FileMode.Create, System.IO.FileAccess.Write, System.IO.FileShare.Read);
			for(int i=0;i<activePage.Length;i++)
			{
				activePage[i] = (ushort)MainForm.Randy.Next();
				dramPage[i] = null;	// Dram.GetPageNumber(activePage[i]);
			}
			clocksToGo = 0;
			HardReset();
		}
		

		/// <summary>
		/// Ask the Xilinx to perform a hard reset
		/// </summary>
		public static void ExecuteOneCrystalCycle()
		{
			// the timing here is totally fudged.
			if(clocksToGo == 0)
			{
				UpdateDmaEvents();
				clocksToGo = 12;	// every cpu cycle, just for fun.
			}
			clocksToGo--;
		}


		/// <summary>
		/// Make the DMA's get a little bit more done.
		/// </summary>
		private static void UpdateDmaEvents()
		{
			if(dmaIdeActive)
			{
				if(dmaIdeCount == 0)
				{
					irqIdent |= 0x01;	// notify that IDE DMA is done.
					dmaIdeActive = false;
				}
				else
				{
					XRam.WriteWord(dmaIdeDestination, Ide.PIO);
					dmaIdeDestination+=2;
					dmaIdeCount--;
				}
			}
			
			if(dmaMp3Active)
			{
				if(dmaMp3Count == 0)
				{
					irqIdent |= 0x02;	// notify that MP3 DMA is done.
					dmaMp3Active = false;
				}
				else
				{
					ushort theData = XRam.ReadWord(dmaMp3Source);
					dmaMp3Source+=2;
					dmaMp3Count--;
				}
			}
		}
		

		/// <summary>
		/// NOTE: Returns TRUE if /INT0 is asserted!!
		/// </summary>
		public static bool INT0
		{
			get
			{
				return ((irqIdent & irqMask) != 0) ? true : false;
			}
		}

		/// <summary>
		/// Ask the Xilinx to perform a hard reset
		/// </summary>
		public static void HardReset()
		{
			wasClock = true;
			wasData = true;
			
			irqIdent = 0;		// no dma's complete.
			irqMask = 0;		// nothing can cause an interrupt.
			
			dmaMp3Source = 0;
			dmaMp3Count= 0;
			dmaMp3Active = false;
			
			dmaIdeDestination = 0;
			dmaIdeCount= 0;
			dmaIdeActive = false;
			
			Write(0xff40, 0);	// cause an IDE hard reset...
		}


		/// <summary>
		/// Update the xilinx object with knowledge of the current state of this pin
		/// </summary>
		public static void Clock(bool bit)
		{
			wasClock = bit;
		}


		/// <summary>
		/// Update the xilinx object with knowledge of the current state of this pin
		/// </summary>
		public static void Data(bool bit)
		{
			wasData = bit;
		}
		

		/// <summary>
		/// Write a short to the Xilinx chip (address = 0xf000..0xffff)
		/// </summary>
		public static void Write(ushort address, byte val)
		{
			// Xilinx Hardware Registers...
			switch(address)
			{
				// DRAM_PAGE_CFG
				case 0xff00:
				case 0xff02:
				case 0xff04:
				case 0xff06:
				case 0xff08:
				case 0xff0a:
				case 0xff0c:
				case 0xff0e:
				case 0xff10:
				case 0xff12:
				case 0xff14:
				case 0xff16:
				case 0xff18:
				case 0xff1a:
				case 0xff1c:
				{
					// choose a specific page of DRAM to map to a page in XRAM.
					int i = (address-0xff00)/2;
					ushort newPageNumber = activePage[i];
					newPageNumber = (ushort)((newPageNumber & 0xff00) | val);
					dramPage[i] = Dram.GetPageNumber(newPageNumber);
					activePage[i] = newPageNumber;
					break;
				}
				
				// DRAM_PAGE_CFG
				case 0xff01:
				case 0xff03:
				case 0xff05:
				case 0xff07:
				case 0xff09:
				case 0xff0b:
				case 0xff0d:
				case 0xff0f:
				case 0xff11:
				case 0xff13:
				case 0xff15:
				case 0xff17:
				case 0xff19:
				case 0xff1b:
				case 0xff1d:
				{
					// choose a specific page of DRAM to map to a page in XRAM.
					int i = (address-0xff00)/2;
					ushort newPageNumber = activePage[i];
					newPageNumber = (ushort)((newPageNumber & 0x00ff) | (((ushort)val)<<8));
					dramPage[i] = Dram.GetPageNumber(newPageNumber);
					activePage[i] = newPageNumber;
					break;
				}

				case 0xff22:	// DMA_IDE_DEST (low byte)
					dmaIdeDestination = (ushort)((dmaIdeDestination & 0xff00) | val);
					break;
					
				case 0xff23:	// DMA_IDE_DEST (high byte)
					dmaIdeDestination = (ushort)((dmaIdeDestination & 0x00ff) | (((ushort)val)<<8));
					break;
				
				case 0xff24:	// DMA_IDE_COUNT (low byte)
					dmaIdeCount = (ushort)((dmaIdeCount & 0xff00) | val);
					break;
					
				case 0xff25:	// DMA_IDE_COUNT (high byte)
					dmaIdeCount = (ushort)((dmaIdeCount & 0x00ff) | (((ushort)val)<<8));
					break;
					
				case 0xff28:	// DMA_MP3_SOURCE (low byte)
					dmaMp3Source = (ushort)((dmaMp3Source & 0xff00) | val);
					break;
					
				case 0xff29:	// DMA_MP3_SOURCE (high byte)
					dmaMp3Source = (ushort)((dmaMp3Source & 0x00ff) | (((ushort)val)<<8));
					break;
					
				case 0xff2a:	// DMA_MP3_COUNT LowByte
					dmaMp3Count = (ushort)((dmaMp3Count & 0xff00) | val);
					break;
				
				case 0xff2b:	// DMA_MP3_COUNT HiByte
					dmaMp3Count = (ushort)((dmaMp3Count & 0x00ff) | (((ushort)val)<<8));
					break;
				
				case 0xff40:	// IDE_RESET_BIT
					if((val & 0x01) == 0)	// active low...
						Ide.HardReset();
					break;
					
				case 0xff51:	// IRQ_MASK
					irqMask = val;
					break;
				
				case 0xff58:	// DMA_IDE_GO
					dmaIdeActive = true;
					break;
				
				case 0xff59:	// DMA_MP3_GO
					dmaMp3Active = true;
					break;
				
				case 0xff5c:	// IRQ_DMA_IDE_ACK Flag
					irqIdent &= (byte)(0xff-1);	// Clear IDE DMA Complete
					break;
				
				case 0xff5d:	// IRQ_DMA_MP3_ACK	Flag
					irqIdent &= (byte)(0xff-2);	// Clear MP3 DMA Complete
					break;
				
				case 0xff5e:	// IRQ_MEMCPY_ACK	Flag
					irqIdent &= (byte)(0xff-4);	// Clear MEMCPY DMA Complete
					break;
				
				case 0xFF62:	// IDE_PORT - IDE Feature Register (write)
					Ide.IdeFeatureRegister = val;
					break;
					
				case 0xFF64:	// IDE_PORT - Sector Count (write)
					Ide.SectorCountRegister = val;
					break;
					
				case 0xFF66:	// IDE_PORT - Sector Number (write)
					Ide.SectorNumberRegister = val;
					break;
					
				case 0xFF68:	// IDE_PORT - Cylinder Low (write)
					Ide.CylinderLowRegister = val;
					break;
					
				case 0xff6a:	// IDE_PORT - Cylinder High (write)
					Ide.CylinderHighRegister = val;
					break;
					
				case 0xff6c:	// IDE_PORT - Device/Head (write)
					Ide.DeviceHeadRegister = val;
					break;
					
				case 0xff6e:	// IDE_PORT - Command (write)
					Ide.CommandRegister = val;
					break;
					
				default:	// any remaining addresses in this page are illegal!!!
					System.Diagnostics.Debug.Assert(false);
					break;
			}
		}
		

		/// <summary>
		/// Write a short to the Xilinx chip (address = 0xf000..0xffff)
		/// </summary>
		public static byte Read(ushort address)
		{
			// Xilinx Hardware Registers...
			switch(address)
			{
				case 0xff50:	// IRQ_IDENT
					return irqIdent;
					
				case 0xff7c:	// IDE_PORT - Alt Status (read)
					return Ide.AlternateStatusRegister;

				default:	// any remaining addresses in this page are illegal!!!
					System.Diagnostics.Debug.Assert(false);
					break;
			}
			return 0;
		}
		

		/// <summary>
		/// Last known state of the clock pin.
		/// </summary>
		private static bool wasClock;

		/// <summary>
		/// Last known state of the data pin.
		/// </summary>
		private static bool wasData;


		/// <summary>
		/// Number of bytes to go in DMA between DRAM and STA013 hardware.
		/// </summary>
		private static ushort dmaMp3Count;


		/// <summary>
		/// Where to fetch the next byte of MP3 data from, in XRAM.
		/// </summary>
		private static ushort dmaMp3Source;

		/// <summary>
		/// TRUE if the DMA is taking place.
		/// </summary>
		private static bool dmaIdeActive;

		/// <summary>
		/// Location in XRAM to dma the data from IDE.
		/// </summary>
		private static ushort dmaIdeDestination;
		
		/// <summary>
		/// Number of shorts to DMA.
		/// </summary>
		private static ushort dmaIdeCount;
		
		/// <summary>
		/// TRUE if the DMA is taking place.
		/// </summary>
		private static bool dmaMp3Active;

		/// <summary>
		/// This byte contains a few bits which notify the CPU exactly *why* an interrupt occurred.
		/// Bit 0: DMA transfer from IDE interface completed 
		/// Bit 1: DMA transfer to MP3 decoder completed 
		/// Bit 2: memcpy operation completed 
		/// </summary>
		private static byte irqIdent;
		

		/// <summary>
		/// This byte contains a few bits which control which events are capable of asserting an interrupt on /INT0, to the CPU.
		/// Bit 0: DMA transfer from IDE interface completed 
		/// Bit 1: DMA transfer to MP3 decoder completed 
		/// Bit 2: memcpy operation completed 
		/// </summary>
		private static byte irqMask;

		
		/// <summary>
		/// Keep track of page-map between Dram and XRam.
		/// </summary>
		private static PageOfDram[] dramPage = new PageOfDram[15];
		

		/// <summary>
		/// Keep track of the current DRAM page# for a specific page.
		/// </summary>
		private static ushort[] activePage = new ushort[15];


		/// <summary>
		/// This is used to figure out when the next DMA update occurs.
		/// </summary>
		private static int clocksToGo;

		/// <summary>
		/// Read a byte of Dram
		/// </summary>
		public static byte ReadDram8(ushort address)
		{
			PageOfDram thePage = dramPage[address >> 12];
			System.Diagnostics.Debug.Assert(thePage != null);	// MP3 player probably performed a read from an uninitialized pointer!
			return thePage.Read8((ushort)(address & 0x0fff));
		}


		/// <summary>
		/// Read a byte of Dram
		/// </summary>
		public static byte ReadDram8(ushort thePage, ushort offset)
		{
			System.Diagnostics.Debug.Assert((thePage>=0)&&(thePage<8192));
			System.Diagnostics.Debug.Assert((offset>=0)&&(offset<4096));
			return Dram.GetPageNumber(thePage).Read8(offset);
		}


		/// <summary>
		/// Write a byte of Dram
		/// </summary>
		public static void WriteDram8(ushort address, byte val)
		{
			PageOfDram thePage = dramPage[address >> 12];
			System.Diagnostics.Debug.Assert(thePage != null);	// MP3 player probably performed a write to an uninitialized pointer!
			thePage.Write8((ushort)(address & 0x0fff), val);
		}

		/// <summary>
		/// Returns TRUE if the XRam address passed in, maps to a legal DRAM Page.
		/// </summary>
		public static bool IsLegalPage(ushort address)
		{
			return (dramPage[address >> 12] != null);
		}

		/// <summary>
		/// Returns the DRAM Page# of the specified address.
		/// For use by the DRAM Memory Form only!!!
		/// NOTE: The address passed in must already have its' page mapped.
		/// </summary>
		public static ushort PeepDramPageNumber(ushort address)
		{
			System.Diagnostics.Debug.Assert(IsLegalPage(address));
			return dramPage[address >> 12].PageNumber;
		}

		/// <summary>
		/// The file which will contain the data sent to the STA013.
		/// </summary>
		private static System.IO.FileStream writer;
	}
}
