using System;

namespace _8052Sim
{
	/// <summary>
	/// Static object.  Simulation is the object which executes a simulation of the hardware.
	/// </summary>
	public class Simulation
	{
		/// <summary>
		/// The timer which causes cycles to execute in the simulation.
		/// </summary>
		private static System.Threading.Thread simulationThread;

		
		/// <summary>
		/// Perform basic initialization for this module.
		/// </summary>
		public static void Startup()
		{
			System.Threading.ThreadStart myThreadStart = new System.Threading.ThreadStart(SimulationThread);
			simulationThread = new System.Threading.Thread(myThreadStart);
			simulationThread.Name = "8051 Simulation Thread";
			simulationThread.Priority = System.Threading.ThreadPriority.Lowest;
			simulationThread.Start();
		}


		/// <summary>
		/// Call to destroy the thread, etc.
		/// </summary>
		public static void Dispose()
		{
			simulationThread.Abort();
		}
		

		/// <summary>
		/// Whether we should run some more of the simulation on next call to ExecuteMoreSimulation()
		/// </summary>
		private static volatile bool isRunning = false;

		/// <summary>
		/// Used to detect if we just returned to the IDE, and therefore need to re-capture values for display.
		/// </summary>
		private static volatile bool wasRunning = false;
		

		/// <summary>
		/// This thread is constantly executing the simulation.  
		/// </summary>
		private static void SimulationThread()
		{
			System.Diagnostics.Debugger.Log(0, "Simulation", "Simulation Thread = "+System.Threading.Thread.CurrentThread.Name+"\n");
			while(true)
			{
//				System.Diagnostics.Debugger.Log(0, "Simulation", "#");
				ExecuteCommands();
				ExecuteMoreSimulation();
				
				System.Threading.Thread.Sleep(0);	// give the system a bit of time.
			}
		}
		
		private static void ExecuteCommands()
		{
			CommandInterface command;
			do
			{
				command = CommandFifo.GetNextCommand();
				if(command != null)
				{
					switch(command.Command)
					{
						case CommandBase.CommandType.Run:		Run();			break;
						case CommandBase.CommandType.Break:		Break();		break;
						case CommandBase.CommandType.StepInto:	StepInto();		break;
						case CommandBase.CommandType.StepOver:	StepOver();		break;
						case CommandBase.CommandType.StepOut:	StepOut();		break;
						case CommandBase.CommandType.Restart:	Restart();		break;

						default:
							System.Diagnostics.Debugger.Log(0, "ExecuteCommand", "No idea what command this is!\n");
							System.Diagnostics.Debug.Assert(false);
							break;
					}
				}
			}
			while(command != null);
		}

		/// <summary>
		/// Execute a bit more of the simulation every time this event executes.
		/// </summary>
		public static void ExecuteMoreSimulation()
		{
			bool justStoppedRunning = false;
			
			if(isRunning)
			{
				int todo = 1;	// number of xtal-cycles per 1/1000th of a second, which need to be executed.
				while(todo != 0)
				{
					if(ExecuteOneCrystalCycleOrUntilBreakpoint())
					{
						isRunning = false;
						justStoppedRunning = true;
						break;
					}
					todo--;
				}
			}
			else
			{
				if(wasRunning)
				{
					justStoppedRunning = true;
				}
			}
			wasRunning = isRunning;
			
			if(justStoppedRunning)
			{
				// We just stopped running!!!
				// re-capture any values which are on display!
				MainForm.SimulationStateMenu.Text = "[AT BREAKPOINT]";
				MainForm.generalPurposeRegsForm.CaptureState();
				MainForm.internalMemoryForm.CaptureState();
				MainForm.timerCounterRegistersForm.CaptureState();
				MainForm.terminalWindowForm.CaptureState();
				MainForm.serialPowerRegistersForm.CaptureState();
				MainForm.programStatusWordForm.CaptureState();
				MainForm.globalVariablesForm.CaptureState();
				MainForm.dramMemoryForm.CaptureState();
				DestroyBreakpointsToDeleteOnBreak();
				MainForm.SourceViewJumpToPC();
			}
		}

		/// <summary>
		/// Perform the same actions as a hard-reset would.
		/// </summary>
		private static void Restart()
		{
			System.Diagnostics.Debugger.Log(0, "Simulation", "--RESTART--\n");

			Break();
			MainForm.SimulationStateMenu.Text = "[AT BREAKPOINT]";
			Cpu8052.HardReset();
			MainForm.SourceViewJumpToPC();
		}

		/// <summary>
		/// User just asked that we stop the simulation at the current instruction
		/// </summary>
		private static void Break()
		{
			System.Diagnostics.Debugger.Log(0, "Simulation", "--BREAK--\n");
			isRunning = false;
		}
		
		/// <summary>
		/// When the simulation stops, all temporary breakpoints need destroyed.
		/// </summary>
		private static void DestroyBreakpointsToDeleteOnBreak()
		{
			foreach(Breakpoint b in allBreakpointsToDeleteOnBreak)
				BreakpointManager.RemoveBreakpoint(b.Address);
			allBreakpointsToDeleteOnBreak.Clear();
		}
		
		/// <summary>
		/// Keeps a collection of all breakpoints which wished to die when the simulation stops.
		/// </summary>
		private static volatile System.Collections.ArrayList allBreakpointsToDeleteOnBreak = new System.Collections.ArrayList();	// Breakpoint objects

		/// <summary>
		/// User just requested that we step over the current instruction
		/// </summary>
		private static void StepOver()
		{
			System.Diagnostics.Debugger.Log(0, "Simulation", "--STEPOVER--\n");

			// stop if we fall out the bottom of the instruction
			Breakpoint b = BreakpointManager.AddFlashBreakpoint(
				(Cpu8052.AddressOverInstruction & 0xffff)
			);
			allBreakpointsToDeleteOnBreak.Add(b);

			// stop if we fall out the bottom of the instruction
			b = BreakpointManager.AddFlashBreakpoint(
				(Cpu8052.AddressOverInstruction & 0xffff) + 0x10000
			);
			allBreakpointsToDeleteOnBreak.Add(b);

			Run();

//			System.Diagnostics.Debugger.Log(0, "Simulation", "--END STEPOVER--\n");
		}
		
		/// <summary>
		/// User just requested that we step over the current instruction
		/// </summary>
		private static void StepOut()
		{
			System.Diagnostics.Debugger.Log(0, "Simulation", "--STEPOUT--\n");

			Run();

			breakWhenSPBelow = Cpu8052.Sp;
			breakWhenSPBelowActive = true;

//			System.Diagnostics.Debugger.Log(0, "Simulation", "--END STEPOUT--\n");
		}
		
		/// <summary>
		/// User just requested that we step into the current instruction
		/// </summary>
		private static void StepInto()
		{
			System.Diagnostics.Debugger.Log(0, "Simulation", "--STEPINTO--\n");

			// stop if we pop out inside the instruction (eg. inside an acall)
			Breakpoint b = BreakpointManager.AddFlashBreakpoint(
				(Cpu8052.AddressIntoInstruction & 0xffff)
			);
			allBreakpointsToDeleteOnBreak.Add(b);

			// stop if we pop out inside the instruction (eg. inside an acall)
			b = BreakpointManager.AddFlashBreakpoint(
				(Cpu8052.AddressIntoInstruction & 0xffff) + 0x10000
			);
			allBreakpointsToDeleteOnBreak.Add(b);

			// or if we fall out the bottom of the instruction!! (like a mov instruction)
			b = BreakpointManager.AddFlashBreakpoint(
				(Cpu8052.AddressOverInstruction & 0xffff)
			);
			allBreakpointsToDeleteOnBreak.Add(b);

			// or if we fall out the bottom of the instruction!! (like a mov instruction)
			b = BreakpointManager.AddFlashBreakpoint(
				(Cpu8052.AddressOverInstruction & 0xffff) + 0x10000
			);
			allBreakpointsToDeleteOnBreak.Add(b);

			Run();
		}

		/// <summary>
		/// Start running the simulation...
		/// </summary>
		private static void Run()
		{
//			System.Diagnostics.Debugger.Log(0, "Simulation", "--RUN--\n");

			MainForm.SimulationStateMenu.Text = "[RUNNING]";
			isRunning = true;
			breakWhenSPBelowActive = false;
		}

		
		/// <summary>
		/// Execute the simulation until a breakpoint is found, or until one 
		/// Crystal cycle completes.
		/// Returns TRUE if a breakpoint was hit, FALSE if one cycle executed.
		/// </summary>
		public static bool ExecuteOneCrystalCycleOrUntilBreakpoint()
		{
			ExecuteOneCrystalCycle();
			if(Cpu8052.MayBreak)
			{
				// check for a typical Breakpoint
				if(ActiveBreakpointHere)
					return true;
					
				// check if the SP has popped enough to mean we're out of the subroutine.
				if(breakWhenSPBelowActive)	// only if that mode is active...
					if(Cpu8052.Sp < breakWhenSPBelow)
						return true;
			}
			return false;
		}
		
		private static bool breakWhenSPBelowActive = false;
		private static byte breakWhenSPBelow = 0;
		
		/// <summary>
		/// Returns TRUE if a breakpoint is here
		/// </summary>
		public static bool ActiveBreakpointHere
		{
			get
			{
				Breakpoint b = Flash.ActiveBreakpointAt(Cpu8052.Pc+(Flash.A16?0x10000:0));
				if(b != null)
					return true;
				return false;
			}
		}
		
		/// <summary>
		/// Execute only one cycle of the simulation
		/// </summary>
		public static void ExecuteOneCrystalCycle()
		{
			cycleNumber++;
			Xilinx.ExecuteOneCrystalCycle();
			Cpu8052.ExecuteOneCrystalCycle();
		}

		public static ulong CycleNumber
		{
			get	{	return cycleNumber;	}
		}

		/// <summary>
		/// The CPU cycle number we are currently executing.
		/// </summary>
		private static ulong cycleNumber = 0;	// 64-bit number
	}
}
