using System;

namespace _8052Sim
{
	/// <summary>
	/// This is the External RAM memory space, accessed via MOVX instructions.
	/// NOTE:  0x0000..0x0efff = DRAM (paged).
	/// 0xf000..0xffff = Xilinx Registers.
	/// </summary>
	public class XRam
	{
		/// <summary>
		/// Constructor
		/// </summary>
		public XRam()
		{
		}


		/// <summary>
		/// Read a byte of XRAM Memory.
		/// </summary>
		public static byte ReadByte(ushort address)
		{
			if(address >= 0xf000)
			{
				// Xilinx Hardware Registers...
				return Xilinx.Read(address);
			}
			else
			{
				// Dram Page Windows...
				return Xilinx.ReadDram8(address);
			}
		}


		/// <summary>
		/// Write a byte of XRAM memory.
		/// </summary>
		public static byte WriteByte(ushort address, byte val)
		{
			if(address >= 0xf000)
			{
				// Xilinx Hardware Registers...
				Xilinx.Write(address, val);
			}
			else
			{
				// Dram Page Windows...
				Xilinx.WriteDram8(address, val);
			}
			return 0;
		}


		/// <summary>
		/// Write a short of XRAM Memory (used by Xilinx DMA's)
		/// </summary>
		public static void WriteWord(ushort address, ushort val)
		{
			WriteByte( (ushort)(address+0), (byte)(val) );		// Low Byte
			WriteByte( (ushort)(address+1), (byte)(val>>8) );	// High Byte
		}

		/// <summary>
		/// Read a short of XRAM Memory (used by Xilinx DMA's)
		/// </summary>
		public static ushort ReadWord(ushort address)
		{
			return (ushort)(
				(ushort)(ReadByte( (ushort)(address+0))) + 		// Low Byte
				(ushort)(ReadByte( (ushort)(address+1))) * 256	// High Byte
			);
		}

		/// <summary>
		/// Read a short of XRAM Memory (used by Xilinx DMA's)
		/// </summary>
		public static ulong ReadLong(ushort address)
		{
			return (ulong)(
				(ulong)(ReadByte( (ushort)(address+0))) + 			// Low Byte
				(((ulong)(ReadByte( (ushort)(address+1)))) <<8) +	// High Byte
				(((ulong)(ReadByte( (ushort)(address+2)))) <<16) +	// High Byte
				(((ulong)(ReadByte( (ushort)(address+3)))) <<24) 	// High Byte
			);
		}
		
		/// <summary>
		/// Returns TRUE if accessing this XRam address will not throw.
		/// Accessing an illegal address will cause a throw if no value has
		/// been assigned to the dramController for the page yet.
		/// </summary>
		public static bool IsLegalAddress(ushort address)
		{
			return Xilinx.IsLegalPage(address);
		}
		
	}
}
